/*******************************************************************************
*
* Replication of Tougher et al (2018)
* 	
* 	CLEANING ROUND 1 DATA
*
*   Last updated 11/06/2018 by ST
*
*******************************************************************************/
cd "$master"

/*******************************************************************************
* TABLE OF CONTENTS
*
* 1. MERGING DATA ON PRESENCE OF SKY PROVIDERS AND REPORTED DATES OF TRAINING, SOCIAL 
*	 FRANCHISING START, AND BRANDING START FROM HEALTH PROVIDER SURVEY
*	1.1 Loading data
*	1.2 Copying social franchising activity dates (from provider survey) in intervention 
*		cluster to control triplet members   
*	1.3 Saving data set
*	1.4 Merging to household survey data
* 2. GENERATING VARIABLES FOR STUDY ARMS
*	2.1 Study arms per protocol (i.e. ITT)
*	2.2 Study arms per presence of sky provider
* 3. CREATING NEW UNIQUE ID SO THERE ARE NO DUPLICATED IDS BETWEEN SURVEYS  
* 4. RESHAPING DATA AND SORTING BY BIRTHDATE TO ENSURE THAT DATA FROM MOST 
*	 RECENT BIRTH IS TAKEN FROM THE BIRTH REGISTER
* 5. CREATING ADDITIONAL VARIABLES RELATED TO INTERVENTION START AND CHILD'S BIRTH DATE
* 	5.1 Quarter since birth
*	5.2 Interview date
*	5.3 Franchising start date, and variables indicating whether birth occurred before 
*		or after franchising
* 6. STUDY OUTCOME
*	6.1 ANC Utilisation
*	6.2 ANC Content of care
*	6.3 ANC Knowledge and preparedness
*	6.4 Delivery care utilisation
*	6.5 Recommended delivery care practices
*	6.6 Harmful or ineffective delivery care practices
*	6.7 Delivery care practices that are frequently overused
*	6.8 Disrespect and abuse
*	6.9 Economic consequences
*	6.10 Postpartum care
*	6.11 Newborn content of care
*	6.12 Neonatal health
*	6.13 Breastfeeding
*	6.14 Family planning
* 	6.15 Setting outcomes so that they are consistently 'good' or 'bad' within a family
* 	6.16 Labelling outcomes
* 7. COVARIATES 
*	7.1 Generating covariates
*	7.2 Labelling covariates variables
* 8. ADDITIONAL LABELLING
* 9. SAVING DATA - ONE OBSERVATION PER ELIGIBLE WOMAN (MOST RECENT DELIVERY)
* 10. RESHAPING DATA FOR ALTERNATIVE MORTALITY INDICATORS (ONE OBSERVATION PREGNANCY
* 11. UPDATING VARIABLES FOR LONG DATA SET
*	11.1 Birth date and interval between birth date and interview
*	11.2 PP3 outcome variables 
* 12. SAVING LONG DATA SET (ONE OBSERVATION PER PREGNANCY)
*
********************************************************************************/

/* SECTION 1 - 	MERGING DATA ON PRESENCE OF SKY PROVIDERS AND REPORTED DATES OF TRAINING, 
				SOCIAL FRANCHISING START, AND BRANDING START FROM HEALTH PROVIDER 
				SURVEY */ 
** 1.1 Loading data 
use "build\input\cluster_sky.dta", clear

/* 1.2 Copying social franchising activity dates in intervention cluster to control 
	triplet members (from provider survey)  */
gsort +triplet -study_arm
by triplet: replace start_whp_training=start_whp_training[_n-1] if start_whp_training==. & study_arm==2
by triplet: replace start_whp_training=start_whp_training[_n-2] if start_whp_training==. & study_arm==1
by triplet: replace start_selfreport=start_selfreport[_n-1] if start_selfreport==. & study_arm==2
by triplet: replace start_selfreport=start_selfreport[_n-2] if start_selfreport==. & study_arm==1
by triplet: replace start_branded=start_branded[_n-1] if start_branded==. & study_arm==2
by triplet: replace start_branded=start_branded[_n-2] if start_branded==. & study_arm==1
rename sky sky_actual
save "build\temp\cluster_sky_triplet.dta", replace

* 1.3 Saving data set
use "build\input\hhsurveydatacompiled.dta", clear

* 1.4 Merging to household survey data
rename i04 id_cluster_sbdhi
merge m:1 id_cluster_sbdhi using "build\temp\cluster_sky_triplet.dta"

drop if _merge==2
drop _merge

* SECTION 2 - GENERATING VARIABLES FOR STUDY ARMS
 
** 2.1 Study arms per protocol (i.e. ITT)
tabulate study_arm, generate(arm_)
rename arm_2 internal
rename arm_1 external
rename arm_3 intervention

** 2.2 Study arms per presence of sky provider
g truearm=.
la var truearm "Study arm based on actual presence of a sky provider"
la def truearmLabs 1 "External" 2 "Internal" 3 "Control"
la val truearm truearmLabs

*** 1=External Controls
replace truearm=1 if external==1 & external!=.

*** 2=Internal Controls
replace truearm=2 if internal==1 & sky_actual==1 & internal!=.
replace truearm=2 if intervention==1 & sky_actual==1 & intervention!=.
replace truearm=2 if internal==1 & sky_actual==2 & internal!=.
	/* The last group (i.e. internal controls where there was reported a sky provider),
		are kep as internal control as this was a data collection error as per 
		email correspondence with Varun Dutt. */

		
*** 3= Intervention clusters
replace truearm=3 if intervention==1 & sky_actual==2 & intervention!=.

tabulate truearm, generate (truearm)
rename truearm1 trueexternal
rename truearm2 trueinternal
rename truearm3 trueintervention

* SECTION 3 - CREATING NEW UNIQUE ID SO THERE ARE NO DUPLICATED IDS BETWEEN SURVEYS  
sort uid
gen temp1 = _n
gen temp2 = 1 
egen uid2 = concat(uid temp1 temp2)
drop temp1 temp2 

la var uid2 "Unique identifier"

/* SECTION 4 - RESHAPING DATA AND SORTING BY BIRTHDATE TO ENSURE THAT DATA FROM MOST 
	RECENT BIRTH IS TAKEN FROM THE BIRTH REGISTER */
	
local birthVars q514 q515 q516 q517 q519 q520 q522 ///
	q523 q524 q525a q525b q526a q526b q527 q528 q529


** Creating a copy of wide birth register variables for mortality vars
foreach x in `birthVars'{ 
	forvalues i = 1(1)9 {
		gen `x'_cpy_`i' = `x'_0`i'
		}  
	gen `x'_cpy_10 = `x'_10
	}
** Labelling newly created variables
foreach x in `birthVars'{ 
	forvalues i = 1(1)9 {
		local lab "Copy of `x'_0`i'"
		display "`lab'"
		la var `x'_cpy_`i' "`lab'"
		}  
	local lab "Copy of `x'_10"
	display "`lab'"
	la var `x'_cpy_10 "`lab'"
	}

** Renaming variables so that reshape cmnd will work
rename q521_mm_ q521_mm_1
rename q521_m_a q521_mm_2
rename q521_m_b q521_mm_3
rename q521_m_c q521_mm_4
rename q521_m_d q521_mm_5
rename q521_m_e q521_mm_6
rename q521_m_f q521_mm_7
rename q521_m_g q521_mm_8
rename q521_m_h q521_mm_9
rename q521_m_i q521_mm_10

rename q521_yy_ q521_yy_1
rename q521_y_a q521_yy_2
rename q521_y_b q521_yy_3
rename q521_y_c q521_yy_4
rename q521_y_d q521_yy_5
rename q521_y_e q521_yy_6
rename q521_y_f q521_yy_7
rename q521_y_g q521_yy_8
rename q521_y_h q521_yy_9
rename q521_y_i q521_yy_10

** Creating a copy of wide birth register variables for mortality vars
foreach x in q521_mm q521_yy { 
	forvalues i = 1(1)9 {
		gen `x'_cpy_`i' = `x'_`i'
		}  
	gen `x'_cpy_10 = `x'_10	
	}
** Labelling newly created variables
foreach x in q521_mm q521_yy { 
	forvalues i = 1(1)9 {
		local lab "Copy of `x'_0`i'"
		display "`lab'"
		la var `x'_cpy_`i' "`lab'"
		}  
	local lab "Copy of `x'_10"
	display "`lab'"
	la var `x'_cpy_10 "`lab'"
	}
** Renaming variables so that reshape cmnd will work
foreach x in `birthVars'{ 
	forvalues i = 1(1)9 {
		rename `x'_0`i' `x'_`i'
		}  
	}
	
** Reshaping data
reshape long q514_ q515_ q516_ q517_ q519_ q520_ q521_mm_ q521_yy_ q522_ q523_ ///
	q524_ q525a_ q525b_ q526a_ q526b_ q527_ q528_ q529_, i(uid2) j(temp)

foreach x in `birthVars'{ 
	rename `x'_ `x'
	}

rename q521_mm_ q521_mm
rename q521_yy_ q521_yy
	
** Variable for birthdate of most recent birth
g birth_m=q521_mm
g birth_y=q521_yy
g birth_date=ym(birth_y, birth_m)
format birth_date %tm
g birth_date_temp=mdy(birth_m, 1, birth_y)
format birth_date_temp %td
g birth_qrt=qofd(birth_date_temp)
format birth_qrt %tq
g birth_hlf=hofd(birth_date_temp)
format birth_hlf %th

** labelling birth register variables
la var q514 "Pregnancy history number"
la var q515 "Single or multiple pregnancy"
la var q516 "Baby born alive, born dead, or lost before birth?"
la var q517 "Did that baby cry, move, or breathe when it was born?"
la var q519 "Child's sex"
la var q520 "Place of delivery"
la var q521_mm "Month of birth"
la var q521_yy "Year of birth"
la var q522 "Is the child still alive"
la var q523 "If born alive and still living, how old was child at last birthday"
la var q524 "If born alive and still living, is the child still living with eligible woman"
la var q525a "If dead, how old was the child when they died - months, days or years"
la var q525b "If dead, how old was the child when they died - number"
la var q526a "If born dead or lost before birth - in what month did the pregnancy end"
la var q526b "If born dead or lost before birth - in what year did the pregnancy end"
la var q527 "If born dead or lost before birth - how many months did this pregnancy last"
la var q528 "If born dead or lost before birth - did you or someone else do something to end this pregnancy"
la var q529 "Were there any pregnancies before this one"

** Dropping if birth_date is missing 
drop if birth_date==.

** Keeping most recent birth
sort uid2 birth_date 
by uid2: keep if _n==_N

drop temp

** Dropping observations where birth was before 2013
drop if birth_date<tm(2013m1)

* SECTION 5 - CREATING ADDITIONAL VARIABLES
** 5.1 quarter since birth
gen qrt_since_birth =.
replace  qrt_since_birth = 220 - birth_qrt
la var qrt_since_birth "Number of quarters between interview date and birth date"

** 5.2 Interview date
g interview_m=i11_mm
g interview_y=i11_yyyy
g interview_date=ym(interview_y, interview_m)
format interview_date %tm	
la var interview_date "Date of interview"

/* 5.3 Franchising start date, and variables indicating whether birth occurred before 
	or after franchising */

*** Flagging one observation per cluster 
sort id_cluster_sbdhi uid
by id_cluster_sbdhi: gen nCluster=_n==1
tab nCluster
la var nCluster "Flagging 1 observation per cluster"

*** Checking number of clusters without a start date
tab1 start_whp_training start_selfreport start_branded if nCluster==1, mi

*** Getting list of cencus codes for intervention arm clusters with missing training date
tab clustercensus_code if start_whp_training==. & study_arm==3

*** Inputing start date from WHP file
merge m:1 clustercensus_code using "build\input\SkyTrainDates", keepusing(training_date sky_health sky_care)

drop if _merge==2
drop _merge

*** Copying sky_care & sky_health vars so that they are non-missing for every observation 
bysort triplet (sky_care): replace sky_care=sky_care[1]
	
bysort triplet (sky_health): replace sky_health=sky_health[1]
	
*** Getting training_date into monthly format (to be consistent with start_whp_training format)
gen training_month =  mofd(training_date)
format training_month %tm
la var training_month "Month of training intervention triplet according to WHP"

gen start_whp_training_ITT=start_whp_training
format start_whp_training_ITT %tm
la var start_whp_training_ITT "Date of training start in intervention triplet according to WHP"

replace start_whp_training_ITT = training_month if start_whp_training==.

*** Showing that intervention arm clusters are no longer missing the training date  
tab clustercensus_code if start_whp_training_ITT==. & study_arm==3

*** Copying start dates to other triplet members
bysort  triplet (start_whp_training_ITT): gen nObsTemp=_n
by triplet: replace  start_whp_training_ITT = start_whp_training_ITT[1] if start_whp_training==.

drop nObsTemp

*** Checking range of training dates
tab start_whp_training_ITT if nCluster==1 

*** Generating variable for treatment quarter
gen start_whp_training_ITT_qrt = qofd(dofm(start_whp_training_ITT))
format start_whp_training_ITT_qrt %tq
la var start_whp_training_ITT_qrt "Quarter of training date"

*** Generating variable for base/end with triplet-specific dates
gen after_ITT = 0 if birth_qrt < start_whp_training_ITT_qrt & birth_qrt !=.
recode after_ITT . = 1 if birth_qrt >= start_whp_training_ITT_qrt & birth_date !=.
la var after_ITT "Birth occurred after WHP-reported training in intervention triplet member"
la def after_ITTLabs 0 "Before" 1 "After"
la val after_ITT after_ITTLabs

*** For robustness checks -> using self-reported date for start date
gen start_selfreport_qrt = qofd(dofm(start_selfreport))
recode start_selfreport_qrt (.=0) 
format start_selfreport_qrt %tq
la var start_selfreport_qrt "Quarter of self-reported start date in intervention triplet member"

*** For robustness checks -> using self-reported branding start date
gen start_selfbrand_qrt = qofd(dofm(start_branded))
recode start_selfbrand_qrt (.=0) 
format start_selfbrand_qrt %tq
la var start_selfbrand_qrt "Quarter of relf-reported branding start date in intervention triplet member"


*** After variable - for robsutness checks
gen after_selfreport = 0 if birth_qrt < start_selfreport_qrt & birth_qrt !=.
recode after_selfreport . = 1 if birth_qrt >= start_selfreport_qrt & birth_date !=.	
la var after_selfreport "Birth occurred after self-reported training date in intervention triplet member"
la def after_selfreportLabs 0 "Before" 1 "After"
la val after_selfreport after_selfreportLabs

gen after_selfbrand = 0 if birth_qrt < start_selfbrand_qrt & birth_qrt !=.
recode after_selfbrand . = 1 if birth_qrt >= start_selfbrand_qrt & birth_date !=.	
la var after_selfbrand "Birth occurred after self-reported branding in intervention triplet member"
la def after_selfbrandLabs 0 "Before" 1 "After"
la val after_selfbrand after_selfbrandedLabs

*** Interaction of after and intervention
gen aftXint = after_ITT*intervention
la var aftXint "Birth occurred after start of franchising and in an intervention cluster"

*** Creating variable for 9 months after training
gen start_whp_training_9mths = start_whp_training_ITT + 9
	format start_whp_training_9mths %tm
	la var start_whp_training_9mths "9 months after WHP-reported training"

gen start_whp_training_9mthsQrt = qofd(dofm(start_whp_training_9mths))	
	format start_whp_training_9mthsQrt %tq
	la var start_whp_training_9mthsQrt "Quarter of 9 months after WHP-reported training"

gen after_9mths_ITT = 0 if birth_qrt < start_whp_training_9mthsQrt & birth_qrt !=.
	recode after_9mths_ITT . = 1 if birth_qrt >= start_whp_training_9mthsQrt & birth_date !=.
	la var after_9mths_ITT "Birth occurred 9 months after WHP-reported training in intervention triplet member"
	la val after_9mths_ITT after_ITTLabs


*** For robustness checks
gen aftXintRob1 = after_ITT*trueintervention
gen aftXintRob3 = aftXint
gen aftXintRob4 = aftXint
gen aftXintRob5 = after_selfbrand*trueintervention

*** For FWER 
tab id_cluster_sbdhi, gen(id_cluster_sbdhi_)

* SECTION 6 - STUDY OUTCOME

** 6.1 ANC UTILIZATION, Variables: anc1_var

*** anc1_3visits = Received at least three ANC visits (%)
gen anc1_3visits=1 if q627>=3
replace anc1_3visit=0 if q627<3
replace anc1_3visit=0 if q626==2
replace anc1_3visit=. if q626==. & q627==.
tab anc1_3visit

*** anc1_tri = Received ANC visit in first trimester (%)-Women who had NO ANC during this preg. coded 0
gen anc1_tri=1 if q630==0 | q630==1 | q630==2 | q630==3
replace anc1_tri=0 if q630>=4
replace anc1_tri=0 if q626==2
replace anc1_tri=. if q626==.
tab anc1_tri

*** anc1_numvis = Number of ANC consultations (visits)
gen anc1_numvis=q627
replace anc1_numvis=0 if q626==2
tab anc1_numvis

*** anc1_ashavis = Woman received visit from ASHA during pregnancy (%)
gen anc1_ashavis=(q625==1) if q625!=.
tab anc1_ashavis

** 6.2 ANC CONTENT OF CARE, Variables: anc2_var

*** anc2_tet = Effective protection with tetanus toxoid vacc (%) [2 or more doses in the past three years]

*** Set variable to missing to start
gen anc2_tet = .

*** Replace with a 1 if woman received 2+ injections during last pregnancy
replace anc2_tet = 1 if q649==1 & q650>=2 & q650!=.

*** Replace with a 1 if woman received 1 injection during last pregnancy and has had an injection previously, with most recent in last 3 years
replace anc2_tet = 1 if q649==1 & q650==1 & q651==1 & q653<=3

*** Replace with a 1 if woman has had 2+ injections previous to current pregnancy, with most recent in last 3 years
replace anc2_tet = 1 if q651==1 & q652>=2 & q652<97 & q653<=3

*** Replace with a 0 if woman had 1 or no injections during most recent pregnancy and none prior
replace anc2_tet = 0 if (q649==2 |(q649==1 & q650==1)) & q651==2

*** Replace with a 0 if woman had 1 or no injections during most recent pregnancy and all prior injections were >3 years earlier
replace anc2_tet = 0 if (q649==2 |(q649==1 & q650==1)) & q651==1 & q653>3 & q653<97

*** Replace with a 0 if woman had  no injections during most recent pregnancy only 1 previous injection prior to that
replace anc2_tet = 0 if q649==2 & q651==1 & q652==1

*** Replace with a 0 if woman did not have any ANC during this last pregnancy
replace anc2_tet = 0 if q626==2

*** anc2_irongiv = Received iron supplementation during pregnancy (%)
gen anc2_irongiv=0 if q647==4
replace anc2_irongiv=0 if q626==2
replace anc2_irongiv=1 if q647==1 | q647==2 | q647==3
tab anc2_irongiv

*** anc2_irontake = Took iron supplementation during pregnancy (defined as 100+ days) (%)
** Those who did not receive iron supps and those who did not have any ANC coded as 0
gen anc2_irontake=1 if q648>=100 & q648!=.
replace anc2_irontake=0 if q648<100 & q648!=. & q648!=98 & q648!=998
replace anc2_irontake=0 if q647==4
replace anc2_irontake=0 if q626==2
tab anc2_irontake

*** anc2_syphres = Test results for syphilis received (%)
gen anc2_syphres=(q640==1) if q640!=.
replace anc2_syphres=0 if q626==2
tab anc2_syphres

*** anc2_abdexam = Abdominal examination during ANC (%)
gen anc2_abdexam=(q636c==1) if q636c!=.
replace anc2_abdexam=0 if q626==2

*** anc2_intwrm = Took a drug for intestinal worms during pregnancy (%)
gen anc2_intwrm=(q644==1) if q644!=.
replace anc2_intwrm=0 if q626==2

*** anc2_mal = Took a drug to prevent malaria
gen anc2_mal=(q645==1) if q645!=.
replace anc2_mal=0 if q626==2

*** anc2_mulbirth = Multiple birth pregnancy detected during ANC (%)
gen anc2_mulbirth=(q635==1) if q635!=.
replace anc2_mulbirth=0 if q634==2
replace anc2_mulbirth=0 if q626==2

*** anc2_content = ANC content of care score of six items [weight, BP, urine, blood, birth preparedness advice, and told about all three danger signs during pregnancy] (index 0 to 1)

*** Women who had no ANC coded as 0

*** weight
gen anc2_weight=(q636a==1) if q636a!=.
replace anc2_weight=0 if q626==2

*** BP
gen anc2_bp=(q637==1) if q637!=.
replace anc2_bp=0 if q626==2

*** urine
gen anc2_urine=(q638==1) if q638!=.
replace anc2_urine=0 if q626==2

*** Blood
gen anc2_blood=(q639==1) if q639!=.
replace anc2_blood=0 if q626==2

*** Birthpreparedness advice
gen anc2_birthprep=(q642==1) if q642!=.
replace anc2_birthprep=0 if q626==2

*** Preg complications
gen anc2_pregcomp=1 if q643a==1 & q643b==1 & q643c==1
replace anc2_pregcomp=0 if q643a==2 | q643b==2 | q643c==2
replace anc2_pregcomp=0 if q626==2

egen anc2_content = rowtotal (anc2_weight anc2_bp anc2_urine anc2_blood anc2_birthprep anc2_pregcomp), missing
replace anc2_content= anc2_content/6

** 6.3. ANC KNOWLEDGE & PREPAREDNESS, Variables: anc3_var

*** anc3_knowpregcomp = Mother knowledge of pregnancy complications (index 0 to 1)[10 complications: headache, blurry vision, absent fetal movement, high bp, edema, convulsions, bleeding, abdominal pain, fever, anaemia]

foreach var of varlist q654_01 q654_02 q654_03 q654_04 q654_05 q654_06 q654_07 q654_08 q654_09 q654_10 q654_11 {
tostring `var', replace force
}

g temp1=strpos(q654_01, "1") | strpos(q654_02, "1") | strpos(q654_03, "1") | strpos(q654_04, "1") | strpos(q654_05, "1") | strpos(q654_06, "1") | strpos(q654_07, "1") | strpos(q654_08, "1") | strpos(q654_09, "1") | strpos(q654_10, "1") | strpos(q654_11, "1")
g temp2=strpos(q654_01, "2") | strpos(q654_02, "2") | strpos(q654_03, "2") | strpos(q654_04, "2") | strpos(q654_05, "2")| strpos(q654_06, "2") | strpos(q654_07, "2") | strpos(q654_08, "2") | strpos(q654_09, "2") | strpos(q654_10, "2") | strpos(q654_11, "2")
g temp3=strpos(q654_01, "3") | strpos(q654_02, "3") | strpos(q654_03, "3") | strpos(q654_04, "3") | strpos(q654_05, "3")| strpos(q654_06, "3") | strpos(q654_07, "3") | strpos(q654_08, "3") | strpos(q654_09, "3") | strpos(q654_10, "3") | strpos(q654_11, "3")

g temp4=strpos(q654_01, "4") | strpos(q654_02, "4") | strpos(q654_03, "4") | strpos(q654_04, "4") | strpos(q654_05, "4")| strpos(q654_06, "4") | strpos(q654_07, "4") | strpos(q654_08, "4") | strpos(q654_09, "4") | strpos(q654_10, "4") | strpos(q654_11, "4")
g temp5=strpos(q654_01, "5") | strpos(q654_02, "5") | strpos(q654_03, "5") | strpos(q654_04, "5") | strpos(q654_05, "5")| strpos(q654_06, "5") | strpos(q654_07, "5") | strpos(q654_08, "5") | strpos(q654_09, "5") | strpos(q654_10, "5") | strpos(q654_11, "5")
g temp6=strpos(q654_01, "6") | strpos(q654_02, "6") | strpos(q654_03, "6") | strpos(q654_04, "6") | strpos(q654_05, "6")| strpos(q654_06, "6") | strpos(q654_07, "6") | strpos(q654_08, "6") | strpos(q654_09, "6") | strpos(q654_10, "6") | strpos(q654_11, "6")
g temp7=strpos(q654_01, "7") | strpos(q654_02, "7") | strpos(q654_03, "7") | strpos(q654_04, "7") | strpos(q654_05, "7")| strpos(q654_06, "7") | strpos(q654_07, "7") | strpos(q654_08, "7") | strpos(q654_09, "7") | strpos(q654_10, "7") | strpos(q654_11, "7")
g temp8=strpos(q654_01, "8") | strpos(q654_02, "8") | strpos(q654_03, "8") | strpos(q654_04, "8") | strpos(q654_05, "8")| strpos(q654_06, "8") | strpos(q654_07, "8") | strpos(q654_08, "8") | strpos(q654_09, "8") | strpos(q654_10, "8") | strpos(q654_11, "8")
g temp9=strpos(q654_01, "9") | strpos(q654_02, "9") | strpos(q654_03, "9") | strpos(q654_04, "9") | strpos(q654_05, "9")| strpos(q654_06, "9") | strpos(q654_07, "9") | strpos(q654_08, "9") | strpos(q654_09, "9") | strpos(q654_10, "9") | strpos(q654_11, "9")
g temp10=strpos(q654_01, "10") | strpos(q654_02, "10") | strpos(q654_03, "10") | strpos(q654_04, "10") | strpos(q654_05, "10")| strpos(q654_06, "10") | strpos(q654_07, "10") | strpos(q654_08, "10") | strpos(q654_09, "10") | strpos(q654_10, "10") | strpos(q654_11, "10")

g anc3_headache=(temp1==1)
g anc3_vision= (temp2==1)
g anc3_absentfetalmovement= (temp3==1)
g anc3_highbp= (temp4==1)
g anc3_edema= (temp5==1)
g anc3_convulsion= (temp6==1)
g anc3_bleeding= (temp7==1)
g anc3_adbpain= (temp8==1)
g anc3_fever= (temp9==1)
g anc3_anaemia= (temp10==1)

egen anc3_knowpregcomp = rowtotal (anc3_headache anc3_vision anc3_absentfetalmovement anc3_highbp anc3_edema anc3_convulsion anc3_bleeding anc3_adbpain anc3_fever anc3_anaemia), missing
replace anc3_knowpregcomp= anc3_knowpregcomp/10

drop temp1 temp2 temp3 temp4 temp5 temp6 temp7 temp8 temp9 temp10

*** anc3_knowdelcomp = Mother knowledge of signs of delivery complications (index 0 to 1)
foreach var of varlist q701_01 q701_02 q701_03 q701_04 q701_05 q701_06 q701_07 q701_08 q701_09 q701_10 q701_11 {
tostring `var', replace force
}

g temp1=strpos(q701_01, "1") | strpos(q701_02, "1") | strpos(q701_03, "1") | strpos(q701_04, "1") | strpos(q701_05, "1") | strpos(q701_06, "1") | strpos(q701_07, "1") | strpos(q701_08, "1") | strpos(q701_09, "1") | strpos(q701_10, "1") | strpos(q701_11, "1")
g temp2=strpos(q701_01, "2") | strpos(q701_02, "2") | strpos(q701_03, "2") | strpos(q701_04, "2") | strpos(q701_05, "2")| strpos(q701_06, "2") | strpos(q701_07, "2") | strpos(q701_08, "2") | strpos(q701_09, "2") | strpos(q701_10, "2") | strpos(q701_11, "2")
g temp3=strpos(q701_01, "3") | strpos(q701_02, "3") | strpos(q701_03, "3") | strpos(q701_04, "3") | strpos(q701_05, "3")| strpos(q701_06, "3") | strpos(q701_07, "3") | strpos(q701_08, "3") | strpos(q701_09, "3") | strpos(q701_10, "3") | strpos(q701_11, "3")

g temp4=strpos(q701_01, "4") | strpos(q701_02, "4") | strpos(q701_03, "4") | strpos(q701_04, "4") | strpos(q701_05, "4")| strpos(q701_06, "4") | strpos(q701_07, "4") | strpos(q701_08, "4") | strpos(q701_09, "4") | strpos(q701_10, "4") | strpos(q701_11, "4")
g temp5=strpos(q701_01, "5") | strpos(q701_02, "5") | strpos(q701_03, "5") | strpos(q701_04, "5") | strpos(q701_05, "5")| strpos(q701_06, "5") | strpos(q701_07, "5") | strpos(q701_08, "5") | strpos(q701_09, "5") | strpos(q701_10, "5") | strpos(q701_11, "5")
g temp6=strpos(q701_01, "6") | strpos(q701_02, "6") | strpos(q701_03, "6") | strpos(q701_04, "6") | strpos(q701_05, "6")| strpos(q701_06, "6") | strpos(q701_07, "6") | strpos(q701_08, "6") | strpos(q701_09, "6") | strpos(q701_10, "6") | strpos(q701_11, "6")
g temp7=strpos(q701_01, "7") | strpos(q701_02, "7") | strpos(q701_03, "7") | strpos(q701_04, "7") | strpos(q701_05, "7")| strpos(q701_06, "7") | strpos(q701_07, "7") | strpos(q701_08, "7") | strpos(q701_09, "7") | strpos(q701_10, "7") | strpos(q701_11, "7")
g temp8=strpos(q701_01, "8") | strpos(q701_02, "8") | strpos(q701_03, "8") | strpos(q701_04, "8") | strpos(q701_05, "8")| strpos(q701_06, "8") | strpos(q701_07, "8") | strpos(q701_08, "8") | strpos(q701_09, "8") | strpos(q701_10, "8") | strpos(q701_11, "8")
g temp9=strpos(q701_01, "9") | strpos(q701_02, "9") | strpos(q701_03, "9") | strpos(q701_04, "9") | strpos(q701_05, "9")| strpos(q701_06, "9") | strpos(q701_07, "9") | strpos(q701_08, "9") | strpos(q701_09, "9") | strpos(q701_10, "9") | strpos(q701_11, "9")
g temp10=strpos(q701_01, "10") | strpos(q701_02, "10") | strpos(q701_03, "10") | strpos(q701_04, "10") | strpos(q701_05, "10")| strpos(q701_06, "10") | strpos(q701_07, "10") | strpos(q701_08, "10") | strpos(q701_09, "10") | strpos(q701_10, "10") | strpos(q701_11, "10")
g temp11=strpos(q701_01, "11") | strpos(q701_02, "11") | strpos(q701_03, "11") | strpos(q701_04, "11") | strpos(q701_05, "11")| strpos(q701_06, "11") | strpos(q701_07, "11") | strpos(q701_08, "11") | strpos(q701_09, "11") | strpos(q701_10, "11") | strpos(q701_11, "11")

g anc3_delbleed=(temp1==1)
g anc3_deldischarge= (temp2==1)
g anc3_delfever= (temp3==1)
g anc3_delfeetfirst= (temp4==1)
g anc3_delabnormalpos= (temp5==1)
g anc3_delprolonglab= (temp6==1)
g anc3_delretainedplacenta= (temp7==1)
g anc3_delruptureduterus= (temp8==1)
g anc3_delprolapsedcord= (temp9==1)
g anc3_delcordneck= (temp10==1)
g anc3_delconvulsions= (temp11==1)

egen anc3_knowdelcomp= rowtotal (anc3_delbleed anc3_deldischarge anc3_delfever anc3_delfeetfirst anc3_delabnormalpos anc3_delprolonglab anc3_delretainedplacenta anc3_delruptureduterus anc3_delprolapsedcord anc3_delcordneck anc3_delconvulsions)
replace anc3_knowdelcomp= anc3_knowdelcomp/11

drop temp1 temp2 temp3 temp4 temp5 temp6 temp7 temp8 temp9 temp10 temp11

*** Birth preparedness (financial, transport, blood donor, attendant, safe delivery kit) (index 0 to 1)
gen anc3_finance= (q656_01==1 | q656_02==1 | q656_03==1 | q656_04==1 | q656_05==1 | q656_06==1 | q656_07==1)

gen anc3_transport= (q656_01==2 | q656_02==2 | q656_03==2 | q656_04==2 | q656_05==2 | q656_06==2 | q656_07==2)

g anc3_blooddonor= (q656_01==4 | q656_02==4 | q656_03==4 | q656_04==4 | q656_05==4 | q656_06==4 | q656_07==4)

g anc3_birthattendant= (q656_01==5 | q656_02==5 | q656_03==5 | q656_04==5 | q656_05==5 | q656_06==5 | q656_07==5)

g anc3_delkit= (q656_01==11 | q656_02==11 | q656_03==11 | q656_04==11 | q656_05==11 | q656_06==11 | q656_07==11)

egen anc3_birthprep = rowtotal (anc3_finance anc3_transport anc3_blooddonor anc3_birthattendant anc3_delkit), missing
replace anc3_birthprep= anc3_birthprep/5

*** Additional anc outcome - AHSA accompanied woman to ANC
gen anc_ashaaccompany=(q629==1) if q629!=.
	recode anc_ashaaccompany (.=0) if q626==2

*** Rec'd at least one public ANC visit
gen anc_public_any = 0
	recode anc_public_any (0=1) if (q628_01>20 & q628_01<29) | (q628_02>20 & q628_02<29) | ///
		(q628_03>20 & q628_03<29) | (q628_04>20 & q628_04<29) | (q628_05>20 & q628_05<29) | ///
		(q628_06>20 & q628_06<29) | (q628_07>20 & q628_07<29) | (q628_08>20 & q628_08<29) | ///
		(q628_09>20 & q628_09<29) | (q628_10>20 & q628_10<29) | (q628_11>20 & q628_11<29) | ///
		(q628_12>20 & q628_12<29) | (q628_13>20 & q628_13<29) | (q628_14>20 & q628_14<29) | ///
		(q628_15>20 & q628_15<29) | (q628_16>20 & q628_16<29) | (q628_17>20 & q628_17<29) | ///
		(q628_18>20 & q628_18<29) | (q628_19>20 & q628_19<29)

*** Rec'd at least one private ANC visit
gen anc_private_any = 0
	recode anc_private_any (0=1) if (q628_01>40 & q628_01<46) | (q628_02>40 & q628_02<46) | ///
		(q628_03>40 & q628_03<46) | (q628_04>40 & q628_04<46) | (q628_05>40 & q628_05<46) | ///
		(q628_06>40 & q628_06<46) | (q628_07>40 & q628_07<46) | (q628_08>40 & q628_08<46) | ///
		(q628_09>40 & q628_09<46) | (q628_10>40 & q628_10<46) | (q628_11>40 & q628_11<46) | ///
		(q628_12>40 & q628_12<46) | (q628_13>40 & q628_13<46) | (q628_14>40 & q628_14<46) | ///
		(q628_15>40 & q628_15<46) | (q628_16>40 & q628_16<46) | (q628_17>40 & q628_17<46) | ///
		(q628_18>40 & q628_18<46) | (q628_19>40 & q628_19<46)
		
*** Rec'd only public ANC
gen anc_public_only = 0
	recode anc_public_only (0=1) if anc_public_any==1 & anc_private_any!=1
	
*** Rec'd only private ANC
gen anc_private_only = 0
	recode anc_private_only (0=1) if anc_private_any==1 & anc_public_any!=1

	
** 6.4 DELIVERY CARE UTILIZATION, Variables: del1_var

*** del1_fac = Gave birth in a health facility (%)
gen del1_fac=0 if q724==11 | q724==12 | q724==13
replace del1_fac=1 if q724==21 | q724==22 | q724==23 | q724==24 | q724==25 | q724==26 | q724==27 | q724==29 | q724==31 | q724==41 | q724==42 | q724==43 | q724==44 | q724==45

*** Observations coded Other-96-1 Dont Know and the other delivery on route to the hospital-coded as missing values here
replace del1_fac=. if q724==96
tab del1_fac

*** del1_sba = Gave birth with a doctor, nurse or midwife (%)
foreach var of varlist q719_1 q719_2 q719_3 q719_4 q719_5 {
tostring `var', replace force
}

g temp1=strpos(q719_1, "1") | strpos(q719_2, "1") | strpos(q719_3, "1") | strpos(q719_4, "1") | strpos(q719_5, "1") if del1_fac!=.
g temp2=strpos(q719_1, "2") | strpos(q719_2, "2") | strpos(q719_3, "2") | strpos(q719_4, "2") | strpos(q719_5, "2") if del1_fac!=.
g temp3=strpos(q719_1, "3") | strpos(q719_2, "3") | strpos(q719_3, "3") | strpos(q719_4, "3") | strpos(q719_5, "3") if del1_fac!=.
g del1_sba=(temp1==1 | temp2 | temp3==1) if del1_fac!=.
drop temp1 temp2 temp3

*** del1_csec = Had a caesarean section (%)
gen del1_csec=(q726==1) if q726!=.
replace del1_csec=0 if q724==11 | q724==12 | q724==13

*** del1_fac_private = delivered in a private facility
gen del1_fac_private =.
recode del1_fac_private (.=1) if q724==41 | q724==42 | q724==43 | q724==44 | q724==45
recode del1_fac_private (.=0) if q724==11 | q724==12 | q724==13 | q724==21 | q724==22 ///
	| q724==23 | q724==24 | q724==25 | q724==26 | q724==27 | q724==29 | q724==31 | ///
	q724==41 | q724==42 | q724==43 | q724==44 | q724==45

*** del1_fac_public = delivered in a public facility	
gen del1_fac_public =. 
recode del1_fac_public (.=1) if q724==21 | q724==22 | q724==23 | q724==24 | q724==25 ///
	| q724==26 | q724==27 | q724==29
recode del1_fac_public (.=0) if q724==11 | q724==12 | q724==13 | q724==31 | q724==41 ///
	| q724==42 | q724==43 | q724==44 | q724==45

	
** 6.5 RECOMMENDED DELIVERY CARE PRACTISES, Variables del2_var

*** del2_glov = Delivery attendant used gloves (%)
*Note, one observation where 11 coded in q719 and responses for 722 and 723 coded- why? how?
gen del2_glov=(q722==1) if q722!=. & q722!=98
replace del2_glov=0 if q719_1=="11" 

*** del2_soap = Delivery attendant washed hands with soap (%)
gen del2_soap=(q723==1) if q723!=. & q723!=98
replace del2_soap=0 if q719_1=="11" 

*** del2_bp = Woman had her BP measured(%)
gen del2_bp= (q731==1) if q731!=.
replace del2_bp=0 if del1_fac==0

*** del2_mob = Mobility during labour (%)
g del2_mob=(q708==1) if q708!=.

*** del2_oralfluids = Oral fluids during labour (%)
g del2_oralfluids= (q710==1) if q710!=.

*** del2_hrbaby = Heart rate of baby monitored with intermittent or continuous auscultation (%)
gen del2_hrbaby= (q716==1 | q716==2) if q716!=. & q716!=98
replace del2_hrbaby=0 if q714==2

*** del2_nasg= Use of antishock garment
gen del2_nasg= (q718==1) if q718!=.

** 6.6 HARMFUL OR INEFFECTIVE DELIVERY CARE PRACTISES, Variables: del3_var

*** del3_birthpos = Position other than lithotomy during bearing down (%)
gen del3_birthpos=1 if q705==1 & q705!=.
replace del3_birthpos=0 if q705==2 | q705==3 | q705==4 | q705==5 | q705==6
replace del3_birthpos=. if q705==98

*** del3_shave = Shaving pubic hair (%)
gen del3_shave= (q713e==1) if q713e!=.

*** del3_enema = Enema given (%)
gen del3_enema= (q713f==1) if q713f!=.

*** del3_ivlab = Intravenous fluids during labour (%)
gen  del3_ivlab= (q713b==1) if q713b!=.

** 6.7 DELIVERY CARE PRACTICES FREQUENTLY OVER USED, Variables: del4_var	

*** del4_urincath = Urinary catheter (%)
gen del4_urincath=(q713g==1) if q713g!=.

*** del4_epidural = Pain control by epidural analgesia (%)
gen del4_epidural= (q717==1) if q717!=.

*** del4_oxyaug = Oxytocin augmentation (%)
gen del4_oxyaug=(q713a==1) if q713a!=.

*** del4_episiotomy = Episiotomy (%)
gen del4_episiotomy=(q713c==1) if q713c!=.

** 6.8 DISRESPECT AND ABUSE, Variables: del5_var 

*** del5_labsup = Support during labour (%)
g del5_labsup=.
replace del5_labsup=1 if q711==1 & del1_fac==1
replace del5_labsup=0 if q711==2 & del1_fac==1

*** del5_noconsent = Medical procedure performed without consent (%)
gen del5_noconsent= (q733==1 & del1_fac==1) if q733!=. & q724!=96

*** del5_shout = Shouted, scolded or humiliated by health worker (%)
g del5_shout= (q734==1 & del1_fac==1) if q734!=. & q724!=96

*** del5_slap = Slapped, pinched or hit by health worker (%)
g del5_slap= (q735==1 & del1_fac==1) if q735!=. & q724!=96

*** del5_priv = Gave birth with privacy (%)
g del5_priv= (q732==1 & del1_fac==1) if q732!=. & q724!=96

*** del5_refcare = Refused care for inability to pay (%)
g del5_refcare=.
replace del5_refcare=1 if q748==1 & del1_fac==1
replace del5_refcare=0 if q748==2 & del1_fac==1 
replace del5_refcare=0 if q746==2 & del1_fac==1

*** del5_disrespect= Felt disrespected during delivery (%)
g del5_disrespect= (q751==1) if q751!=. & q724!=96

*** del5_held= Held in facility due to inability to pay (%)
g del5_held= (q743==1) if q743!=. & q724!=96


** 6.9 ECONOMIC CONSEQUENCES, Variables: del6_var

*** del6_ delspend = Out-of-pocket spending on delivery care (NRS)
clonevar del6_delspend= q747
replace del6_delspend=. if del6_delspend==999998
replace del6_delspend=. if del6_delspend==99998
replace del6_delspend=0 if q746==2

*** del6_borrow = Borrowed money to pay for delivery care (%)
g del6_borrow= (q749==1) if q749!=.
replace del6_borrow=0 if q746==2

*** del6_hhdebt = Household in debt to pay for delivery care (%)
g del6_hhdebt= (q749==1) if q749!=. & q749!=98
replace del6_hhdebt=0 if q749==1 & q750==1 
replace del6_hhdebt=0 if q749==1 & q750==4
replace del6_hhdebt=. if q750==98
replace del6_hhdebt=0 if q746==2

*** del6_jsy = Received JSY cash incentive (%)
*Based on discussions, this is for all live births, regardless of whether a facility birth or not
gen del6_jsy= (q745==1) if q745!=. & q745!=98 & q724!=96
replace del6_jsy=0 if del1_fac==0

*** Additional outcome for referral
gen del_referral =0 if  q736==2
recode del_referral (.=1) if q739==1

*** Additional outcome for ASHA accompanied delivery
gen del_fac_asha = 0
recode del_fac_asha (0=1) if q729_1==1 | q729_2==1 | q729_3==1 | q729_4==1 | q729_5==1 ///
	| q729_6==1
replace del_fac_asha =. if q724==96
	

** 6.10 POSTPARTUM CARE, , Variables: pp1_var

*** pp1_48hrcheck = Woman received postpartum care within 48 hours of birth
gen pp1_woman48hrcheck=(q752==1) if q752!=.

*** pp1_baby48hrcheck = Newborn received postnatal care within 48 hours of birth
*Here defined as Y if q768<3 (so check took place 0,1,2 days after delivery)-Don't know-99 and 98 coded as 0
gen pp1_baby48hrcheck=1 if q768==0 | q768==1 | q768==2
replace pp1_baby48hrcheck=0 if q768>=3 & q768!=.
	

** 6.11 NEWBORN CONTENT OF CARE, Variables: pp2_var

*** pp2_cleancordcare = Clean cord care (clean instrument to cut and tie the cord, and nothing put on cord) (%)

*** pp2_cleancut = Clean instrument used to cut the cord
 g pp2_cleancut=1 if q778==1 | q778==2
 replace pp2_cleancut=0 if q778==3 | q778==4

*** pp2_cleantie = Clean instrument used to tie the cord
gen pp2_cleantie=1 if q777==1 | q777==2 | q777==4
replace pp2_cleantie=0 if q777==3 | q777==5| q777==96

*** pp2_nothingoncord = Nothing put on cord
 gen pp2_nothingoncord= (q779==2) if q779!=.

*** pp2_cleancordcare = Clean cord care
gen pp2_cleancordcare=1 if pp2_cleancut==1 & pp2_cleantie==1 & pp2_nothingoncord==1
replace pp2_cleancordcare=0 if pp2_cleancut==0 | pp2_cleantie==0 | pp2_nothingoncord==0

*** Thermal care (immediate drying, wrapping, skin to skin, and delayed bathing) (%)--CORE OUTCOME

*** pp2_immeddry = Immediate drying
gen pp2_immeddry=1 if q774<31
replace pp2_immeddry=0 if q774>30 & q774!=.
replace pp2_immeddry=0 if q773==2
replace pp2_immeddry=.  if q774==98

*** pp2_immedwrap = Immediate wrapping
gen pp2_immedwrap=1 if q775==1 & q776<=30
replace pp2_immedwrap=0 if q775==1 & q776>=31 & q776!=. 
replace pp2_immedwrap=. if q775==1 & q776==98
replace pp2_immedwrap=0 if q775==2

*** pp2_delbath = Delayed bathing
gen pp2_delbath=1 if q781==3 | q781==4 | q781==5
replace pp2_delbath=0 if q781==1 | q781==2
*Codes 4 (did not bathe till now) and 5 (did not bathe, child died) coded as 1
* To check
tab q521_mm  q521_yy if q781==4
tab q521_mm  q521_yy if q781==5

*** pp2_immedskin = Generate variable for immediate skin to skin 
gen pp2_immedskin=1 if q772==2
replace pp2_immedskin=0 if q772==1 | q772==3 | q772==4 | q772==5

*** pp2_thermcare = Thermal care
gen pp2_thermcare=1 if pp2_immeddry==1 & pp2_immedwrap==1 & pp2_delbath==1 & pp2_immedskin==1
replace pp2_thermcare=0 if pp2_immeddry==0 | pp2_immedwrap==0 | pp2_delbath==0 | pp2_immedskin==0

*** pp2_babyweighed = Baby weighed at birth (%)
gen pp2_babyweighed= .
	recode pp2_babyweighed (.=1) if q703==1
	recode pp2_babyweighed (.=0) if q703==2

*** pp2_birthregistered = Baby registered and received certificate (%)
gen pp2_birthregistered= (q763==1 & q764==1) if q763!=. & q763!=98 & q764!=.
replace pp2_birthregistered =0 if q763==2
replace pp2_birthregistered =0 if q763==1 & q764==2
	
	
** 6.12 NEONATAL HEALTH, Variables: pp3_var
	
*** pp3_nmr = Neonatal mortality (per 1,000 live births)*Most recent birth, not full sample- see NMR_full sample dofile
gen pp3_nmr=1 if q516==1 & q522==2 & q525a==1 & q525b<=27

replace pp3_nmr=1 if q516==2 & q517==1 & q522==2 & q525a==1 & q525b<=27

*** Not neonatal death if child died on or after 28 days of birth
replace pp3_nmr=0 if q516==1 & q522==2 & q525a==1 & q525b>=28

*** Not neonatal death if child died any number of months after birth
replace pp3_nmr=0 if q516==1 & q522==2 & q525a==2

*** Not neonatal death if child died any number of years after birth
replace pp3_nmr=0 if q516==1 & q522==2 & q525a==3

*** Child born alive and still living
replace pp3_nmr=0 if q516==1 & q522==1

*** Child reported as born dead, showed signs of life at birth, and is still living
replace pp3_nmr=0 if q516==2 & q517==1 & q522==1

*** Child reported as dead, showed signs of life, is now dead, but died 28 or more days after birth
replace pp3_nmr=0 if q516==2 & q517==1 & q522==2 & q525a==1 & q525b>=28

*** Child reported as born dead, showed signs of life at birth, is now dead, died after one month of birth
replace pp3_nmr=0 if q516==2 & q517==1 & q522==2 & q525a==2

*** Child reported as born dead, showed signs of life at birth, is now dead, and died after 1 year of birth
replace pp3_nmr=0 if q516==2 & q517==1 & q522==2 & q525a==3

gen livebirths= (q516==1) if q516!=.

replace livebirths=1 if q516==2 & q517==1 & q516!=.

*** pp3_onedaymort = One-day mortality (per 1,000 live births)
g pp3_onedaymort=1 if q516==1 & q522==2 & q525a==1 & q525b<=1

replace pp3_onedaymort=1 if q516==2 & q517==1 & q522==2 & q525a==1 & q525b<=1

*** Not one day death if child died after 1 days of birth
replace pp3_onedaymort=0 if q516==1 & q522==2 & q525a==1 & q525b>=2

*** Not one day death if child died any number of months after birth
replace pp3_onedaymort=0 if q516==1 & q522==2 & q525a==2

*** Not death if child died any number of years after birth
replace pp3_onedaymort=0 if q516==1 & q522==2 & q525a==3

*** Child born alive and still living
replace pp3_onedaymort=0 if q516==1 & q522==1

*** Child reported as born dead, showed signs of life at birth, and is still living
replace pp3_onedaymort=0 if q516==2 & q517==1 & q522==1

*** Child reported as dead, showed signs of life, is now dead, but died 28 or more days after birth
replace pp3_onedaymort=0 if q516==2 & q517==1 & q522==2 & q525a==1 & q525b>=2

*** Child reported as born dead, showed signs of life at birth, is now dead, died after one month of birth
replace pp3_onedaymort=0 if q516==2 & q517==1 & q522==2 & q525a==2

*** Child reported as born dead, showed signs of life at birth, is now dead, and died after 1 year of birth
replace pp3_onedaymort=0 if q516==2 & q517==1 & q522==2 & q525a==3

*** pp3_birthweight = Birth weight (kg)
clonevar pp3_birthweight= q704b

	
** 6.13 BREASTFEEDING, Variables = pp4_var	

*** pp4_breastfeed = Immediate breastfeeding within one hour of birth (%)
gen pp4_breastfeed= (q784==1 & q785==1) if q784!=. & q785!=. 
replace pp4_breastfeed=0 if q784==2

*** pp4_colostrum = Colostrum given to baby (%)
gen pp4_colostrum= (q786==2) if q786!=.
replace pp4_colostrum =0 if q784==2

*** pp4_breastfeeding3days = Exclusive breastfeeding for three days (%)
g pp4_breastfeed3days= (q787==2) if q787!=.
replace pp4_breastfeed3days=0 if q784==2
	
	
** 6.14 FAMILY PLANNING, Variables: pp5_var
	
*** pp5_modfp = Current modern contraceptive use (%)
gen pp5_modfp=.
recode pp5_modf (.=0) if q611==2 | q612==2

recode pp5_modf (.=1) if q613_01==1 | q613_01==2 | q613_01==3 | q613_01==4 | q613_01==5 | ///
	q613_01==6 | q613_01==7 | q613_01==8
recode pp5_modf (.=1) if q613_02==1 | q613_02==2 | q613_02==3 | q613_02==4 | q613_02==5 | ///
	q613_02==6 | q613_02==7 | q613_02==8
recode pp5_modf (.=1) if q613_03==1 | q613_03==2 | q613_03==3 | q613_03==4 | q613_03==5 | ///
	q613_03==6 | q613_03==7 | q613_03==8
recode pp5_modf (.=1) if q613_04==1 | q613_04==2 | q613_04==3 | q613_04==4 | q613_04==5 | ///
	q613_04==6 | q613_04==7 | q613_04==8
recode pp5_modf (.=1) if q613_05==1 | q613_05==2 | q613_05==3 | q613_05==4 | q613_05==5 | ///
	q613_05==6 | q613_05==7 | q613_05==8
recode pp5_modf (.=1) if q613_06==1 | q613_06==2 | q613_06==3 | q613_06==4 | q613_06==5 | ///
	q613_06==6 | q613_06==7 | q613_06==8
recode pp5_modf (.=1) if q613_07==1 | q613_07==2 | q613_07==3 | q613_07==4 | q613_07==5 | ///
	q613_07==6 | q613_07==7 | q613_07==8	
recode pp5_modf (.=1) if q613_08==1 | q613_08==2 | q613_08==3 | q613_08==4 | q613_08==5 | ///
	q613_08==6 | q613_08==7 | q613_08==8	
recode pp5_modf (.=1) if q613_09==1 | q613_09==2 | q613_09==3 | q613_09==4 | q613_09==5 | ///
	q613_09==6 | q613_09==7 | q613_09==8	
recode pp5_modf (.=1) if q613_10==1 | q613_10==2 | q613_10==3 | q613_10==4 | q613_10==5 | ///
	q613_10==6 | q613_10==7 | q613_10==8
recode pp5_modf (.=1) if q613_11==1 | q613_11==2 | q613_11==3 | q613_11==4 | q613_11==5 | ///
	q613_11==6 | q613_11==7 | q613_11==8
recode pp5_modf (.=1) if q613_12==1 | q613_12==2 | q613_12==3 | q613_12==4 | q613_12==5 | ///
	q613_12==6 | q613_12==7 | q613_12==8

recode pp5_modf (.=0) if q613_01==9 | q613_01==10 | q613_01==11
recode pp5_modf (.=0) if q613_02==9 | q613_02==10 | q613_02==11	
recode pp5_modf (.=0) if q613_03==9 | q613_03==10 | q613_03==11		
recode pp5_modf (.=0) if q613_04==9 | q613_04==10 | q613_04==11		
recode pp5_modf (.=0) if q613_05==9 | q613_05==10 | q613_05==11		
recode pp5_modf (.=0) if q613_06==9 | q613_06==10 | q613_06==11		
recode pp5_modf (.=0) if q613_07==9 | q613_07==10 | q613_07==11		
recode pp5_modf (.=0) if q613_08==9 | q613_08==10 | q613_08==11		
recode pp5_modf (.=0) if q613_09==9 | q613_09==10 | q613_09==11		
recode pp5_modf (.=0) if q613_10==9 | q613_10==10 | q613_10==11		
recode pp5_modf (.=0) if q613_11==9 | q613_11==10 | q613_11==11		
recode pp5_modf (.=0) if q613_12==9 | q613_12==10 | q613_12==11		
		

*** pp5_modfpknowledge=  Knowledge of modern contraceptive methods (%)
recode q601 (2 =0)
recode q602 (2 =0)
recode q603 (2 =0)
recode q604 (2 =0)
recode q605 (2 =0)
recode q606 (2 =0)
recode q607 (2 =0)
recode q610 (2 =0)

g pp5_modfpknowledge=1 if q601==1 | q602==1 | q603==1 | q604==1 | q605==1 | q606==1 | q607==1 | q610==1
replace pp5_modfpknowledge=0 if q601==0 & q602==0 & q603==0 & q604==0 & q605==0 & q606==0 & q607==0 & q610==0

*** Knowledge score -0 to 1 (8 methods of modern contraceptives)
egen pp5_modfpscore = rowtotal (q601 q602 q603 q604 q605 q606 q607 q610), missing
replace pp5_modfpscore= pp5_modfpscore/8


** 6.15 Setting outcomes so that they are consistently 'good' or 'bad' within a family

*** Binary outcomes
local oppositeOutcomes del3_birthpos del3_shave del3_enema del3_ivlab ///
	del4_urincath del4_epidural del4_oxyaug del4_episiotomy ///
	del5_labsup del5_noconsent del5_shout del5_slap del5_priv del5_refcare del5_disrespect del5_held ///
	del6_borrow del6_hhdebt del6_jsy ///
	pp3_nmr pp3_onedaymort 

foreach var of local oppositeOutcomes {
	gen `var'_alt = `var'
	recode `var'_alt (0=1) (1=0)
	}

*** Continuous outcomes
local oppositeOutcomesCont del6_delspend
foreach var of local oppositeOutcomesCont {
	gen `var'_alt = `var'*-1
	}
	
** 6.16 Labelling outcomes
*** ANC1
la var anc1_3visits "Received at least 3 ANC vists"
la var anc1_tri "Received ANC visit in first trimester"
la var anc1_numvis "Number of ANC consultations"
la var anc1_ashavis "Received visit from ASHA "

*** ANC2 - Main
la var anc2_tet "Fully immunised with tetanus toxoid"
la var anc2_irongiv "Received iron supplementation"
la var anc2_irontake "Iron supplementation for 100 days"
la var anc2_syphres "Received test results for syphilis"
la var anc2_abdexam "Abdominal examination during ANC"
la var anc2_intwrm "Received a drug for intestinal worms"
la var anc2_mal "Received a drug to prevent malaria"
la var anc2_mulbirth "Multiple birth pregnancy detected during ANC"
la var anc2_content "ANC content of care score of six items"

*** ANC - Intermediate or additional outcome
la var anc2_birthprep "Advice irth preparedness given during ANC"
la var anc2_blood "Blood taken during ANC"
la var anc2_bp "Blood pressure taken during ANC"
la var anc2_pregcomp "Pregnancy complications explained during ANC"
la var anc2_urine "Urine taken during ANC"
la var anc2_weight "Weight measured during ANC"
	
*** ANC3 - Main
la var anc3_knowpregcomp "Mother knowledge of pregnancy complications index (0 to 1)" 
la var anc3_knowdelcomp "Mother knowledge of pregnancy complications index (0 to 1)"
la var anc3_birthprep "Birth preparedness index (0 to 1)"

*** ANC3 - Intermediate or additional outcomes
la var anc3_absentfetalmovement "Mother's knowledge of pregnancy complication: absent fetal movement"
la var anc3_adbpain "Mother's knowledge of pregnancy complication: abdominal pain"
la var anc3_anaemia "Mother's knowledge of pregnancy complication: anemia"
la var anc3_birthattendant "Birth preparedness: birth attendant"
la var anc3_bleeding "Mother's knowledge of pregnancy complication: bleeding"
la var anc3_blooddonor "Birth preparedness: blood donor"
la var anc3_convulsion "Mother's knowledge of pregnancy complication: convulsion"
la var anc3_delabnormalpos  "Mother's knowledge of delivery complications: abnormal position"
la var anc3_delbleed "Mother's knowledge of delivery complications: bleeding"
la var anc3_delconvulsions  "Mother's knowledge of delivery complications: convulsions"
la var anc3_delcordneck  "Mother's knowledge of delivery complications: cord around neck"
la var anc3_deldischarge "Mother's knowledge of delivery complications: discharge" 
la var anc3_delfeetfirst  "Mother's knowledge of delivery complications: feet first"
la var anc3_delfever  "Mother's knowledge of delivery complications: fever"
la var anc3_delkit "Birth preparedness: safe delivery kit"
la var anc3_delprolapsedcord  "Mother's knowledge of delivery complications: prolapsed cord"
la var anc3_delprolonglab  "Mother's knowledge of delivery complications: prolonged labour"
la var anc3_delretainedplacenta  "Mother's knowledge of delivery complications: retained placenta"
la var anc3_delruptureduterus  "Mother's knowledge of delivery complications: ruptured uterus" 
la var anc3_edema "Mother's knowledge of pregnancy complication: edema"
la var anc3_fever "Mother's knowledge of pregnancy complication: fever"
la var anc3_finance "Birth preparedness: financing"
la var anc3_headache "Mother's knowledge of pregnancy complication: headache"
la var anc3_highbp "Mother's knowledge of pregnancy complication: high blood pressure"
la var anc3_transport "Birth preparedness: transport"
la var anc3_vision "Mother's knowledge of pregnancy complication: vision symptoms"

*** Additional ANC outcomes
la var anc_ashaaccompany "Asha accompanied to ANC" 
la var anc_private_any "At least one ANC visit to a private provider" 
la var anc_private_only "All ANC visits at private providers"
la var anc_public_any "At least one ANC visit to a public provider" 
la var anc_public_only "All ANC visits at public providers"

*** DEL1 - Main
la var del1_fac "Gave birth in a health facility"
la var del1_sba "Gave birth with a doctor, nurse or midwife"
la var del1_csec "Had a caesarean section"

*** DEL 1 - Intermediate or additional outcomes
la var del1_fac_private "Gave birth in a private facility"
la var del1_fac_public "Gave birth in a public facility"

*** DEL2
la var del2_glov "Delivery attendant used gloves"
la var del2_soap "Delivery attendant washed hands with soap"
la var del2_bp "Woman had her BP measured"
la var del2_mob "Mobility during labour"
la var del2_oralfluids "Oral fluids during labour"
la var del2_hrbaby "Heart rate of baby monitored with intermittent or continuous auscultation"
la var del2_nasg "Use of anti-shock garment"

*** DEL3 - Main
la var del3_shave "Shaved pubic hair"
la var del3_enema "Enema given"
la var del3_birthpos "Lithotomy position during labour" 
la var del3_ivlab "Intravenous fluids during labour"

*** DEL3 - Intermediate or additional outcomes
la var del3_birthpos_alt "Did not use lithotomy position during labour" 
la var del3_enema_alt "Enema not given during labour"
la var del3_ivlab_alt "IV fluids not given during labour"
la var del3_shave_alt "Pubic hair not shaved for labour"

*** DEL4 - Main
la var del4_urincath "Urinary catheter"
la var del4_epidural "Pain control by epidural analgesia"
la var del4_oxyaug "Oxytocin augmentation"
la var del4_episiotomy "Episiotomy"

*** DEL4 - Intermediate or additional outcomes
la var del4_epidural_alt "Did not rec'v pain control by epidural analgesia"
la var del4_episiotomy_alt "Did not rec'v episiotomy"
la var del4_oxyaug_alt "Did not rec'v oxytocin augmentation"
la var del4_urincath_alt "Did not rec'v urinary catheter"

*** DEL5 - Main
la var del5_labsup_alt "No support during labour"
la var del5_noconsent "Medical procedure performed without consent"
la var del5_shout "Shouted, scolded or humiliated by health worker"
la var del5_slap "Slapped, pinched or hit by health worker"
la var del5_priv "Gave birth with privacy" 
la var del5_priv_alt "Gave birth without privacy" 
la var del5_refcare "Refused care for inability to pay" 
la var del5_held "Kept in facility for inability to pay"
la var del5_disrespect "Felt disrespected or abused during facility stay"

*** DEL 5 - Intermediate or additional outcomes
la var del5_disrespect_alt "Did not feel disrespected or abused during facility stay"
la var del5_held_alt "Not kept in facility for inability to pay"
la var del5_labsup "Support during labour"
la var del5_noconsent_alt "No procedures performed without consent"
la var del5_refcare_alt "Not refused care for inability to pay"
la var del5_shout_alt "Not shouted, scolded or humiliated by health worker"
la var del5_slap_alt "Not slapped, pinched or hit by health worker"

*** DEL6 - Main
la var del6_delspend "Out-of-pocket spending on delivery care" 
la var del6_borrow "Borrowed money to pay for delivery care"
la var del6_hhdebt "Household in debt to pay for delivery care"
la var del6_jsy_alt "Did not receive JSY cash"

*** DEL6 - Intermediate or additional outcomes
la var del6_borrow_alt "Did not borrow money to pay for delivery care" 
la var del6_delspend_alt "Negative value of out-of-pocket spending on delivery care" 
la var del6_hhdebt_alt "Household did not incurr debt to pay for delivery care"
la var del6_jsy "Rec'd JSY cash"

*** Additional delivery care outcomes
la var del_fac_asha "Asha accompanied women facility for delivery"
la var del_referral "Woman referred to another facility"

*** PP1
la var pp1_woman48hrcheck "Received postpartum care within 48hr of birth" 
la var pp1_baby48hrcheck "Newborn received postnatal care within 48hr of birth"

*** PP2 - main
la var pp2_cleancordcare "Clean cord care"
la var pp2_thermcare "Thermal care"
la var pp2_babyweighed "Baby weighed at birth"
la var pp2_birthregistered "Baby registered and received certificate"

*** PP2 - Intermediate or additional outcomes
la var pp2_cleancut "Clean instrument used to cut the cord"
la var pp2_cleantie "Clean instrument used to tie the cord"
la var pp2_delbath "Delayed bathing"
la var pp2_immeddry "Immediate drying"
la var pp2_immedskin "Immediate skin to skin"
la var pp2_immedwrap "Immediate wrapping"
la var pp2_nothingoncord "Nothing put on cord"

*** PP3 - main
la var pp3_nmr_alt "Neonatal survival"
la var pp3_nmr "Neonatal mortality"
la var pp3_onedaymort_alt  "One day survival"
la var pp3_onedaymort  "One day mortality"
la var pp3_birthweight "Birth weight (KG)"

*** PP3 - Intermediate or additional outcomes
la var livebirths "Baby born alive" 

*** PP4
la var pp4_breastfeed "Immediate breastfeeding within 1hr of birth"
la var pp4_colostrum "Colostrum given to baby"
la var pp4_breastfeed3days "Exclusive breastfeeding for 3 days"

*** PP5 - main
la var pp5_modfp "Current modern contraceptive use"

*** PP5 - intermediate or additional outcomes
la var pp5_modfpknowledge "Knowledge of modern family planning methods"
la var pp5_modfpscore "Knowledge of modern family planning method score"


* SECTION 7 - COVARIATES
** 7.1 Generating covariates
*** hh members
egen hh_member=rownonmiss(q103_01 q103_02 q103_03 q103_04 q103_05 q103_06 q103_07 q103_08 q103_09 q103_10 q103_11 q103_12 q103_13 q103_14 q103_15 q103_16 q103_17 q103_18 q103_19 q103_20)
recode hh_member (0=.)

*** hh_ethnicity = ethnicity-Defined as SC/ST/OBC (1) vs General Caste (0)
gen hh_sc= (q303==1) if q303!=.
g hh_st= (q303==2) if q303!=.
g hh_obc= (q303==3) if q303!=.
g hh_gc= (q303==4) if q303!=.

*** hh_religion= religion
*1= Hindu 2=Muslim 3=Christian 96=Other
tab q301, gen(hh_religion)

gen religion_hindu = 0 if q301!=.
recode religion_hindu (0=1) if q301==1

gen religion_other = 0 if q301!=.
recode religion_other (0=1) if q301!=1 

*** Urban residence
g hh_urban= (urban==2) if urban!=.
g hh_rural= (urban==1) if urban!=.

*** Health insurance status
g hh_insurance= (q306==1) if q306!=. & q306!=98
g hh_noinsurance= (q306==2) if q306!=. & q306!=98

*** Age of eligible woman
g wm_age=.
foreach i of numlist 1/9 {
replace wm_age=q104b_0`i' if q109_0`i'==1
}
foreach i of numlist 10/20 {
replace wm_age=q104b_`i' if q109_`i'==1
}
recode wm_age (2=20)

*** Household head education
g hh_education=0 if q107_01==2
replace hh_education=q108_01 if q107_01==1
recode hh_education (99=.) (98=.)
g hh_ed_none=(hh_education==0) if hh_education!=.
g hh_ed_primary=(hh_education>0 & hh_education<6) if hh_education!=.
g hh_ed_secondary=(hh_education>5 & hh_education<12) if hh_education!=.
g hh_ed_above=(hh_education>11 & hh_education<30) if hh_education!=.
g hh_educ_group=1 if hh_ed_none==1
replace hh_educ_group=2 if hh_ed_primary==1
replace hh_educ_group=3 if hh_ed_secondary==1
replace hh_educ_group=4 if hh_ed_above==1

*** Maternal education
g mat_education=.
foreach i of numlist 1/9 {
replace mat_education=0 if q107_0`i'==2 & q109_0`i'==1
}
foreach i of numlist 10/20 {
replace mat_education=0 if q107_`i'==2 & q109_`i'==1
}
foreach i of numlist 1/9 {
replace mat_education=q108_0`i' if q107_0`i'==1 & q109_0`i'==1
}
foreach i of numlist 10/20 {
replace mat_education=q108_`i' if q107_`i'==1 & q109_`i'==1
}
recode mat_education (99=.)
g mat_ed_none=(mat_education==0) if mat_education!=.
g mat_ed_primary=(mat_education>0 & mat_education<6) if mat_education!=.
g mat_ed_secondary=(mat_education>5 & mat_education<12) if mat_education!=.
g mat_ed_above=(mat_education>11 & mat_education<30) if mat_education!=.

*** parity
g hh_parity=1 if q513==1 & q513!=.
replace hh_parity=2 if q513==2 & q513!=.
replace hh_parity=3 if q513==3 & q513!=.
replace hh_parity=4 if q513==4 & q513!=.
replace hh_parity=5 if q513>=5 & q513!=.

tab hh_parity, gen (parity)

*** multiple birth
g hh_mulbirth= (q515==2) if q515!=.
g hh_singbirth= (q515==1) if q515!=.

*** HH below poverty line- if card was not seen, coded 0
g hh_bpl= (q305==1) if q305!=.
g hh_apl= (q305==2 | q305==3) if q305!=.

*** mattress
g hh_mattress= (q313a==1) if q313a!=.

*** pressure cooker
g hh_pressurecook= (q313b==1) if q313b!=.

*** thermos/casserole
g hh_thermos= (q313c==1) if q313c!=.

*** chair
g hh_chair= (q313d==1) if q313d!=.

*** bed/cot
g hh_bed= (q313e==1) if q313e!=.

*** table
g hh_table= (q313f==1) if q313f!=.

*** almirah/dressing table
g hh_almirah= (q313g==1) if q313g!=.

*** electric fan
g hh_fan= (q313h==1) if q313h!=.

*** radio or transistor
g hh_radio= (q313i==1) if q313i!=.

*** colour tv
g hh_tv= (q313j==1) if q313j!=.

*** vcr/vcd/dvd player
g hh_dvd= (q313k==1) if q313k!=.

*** sewing machine
g hh_sewingmachine= (q313l==1) if q313l!=.

*** mobile telephone
g hh_mobile= (q313m==1) if q313m!=.

*** telephone- any other type
g hh_telephone= (q313n==1) if q313n!=.

*** computer/laptop
g hh_computer= (q313o==1) if q313o!=.

*** refrigerator
g hh_fridge= (q313p==1) if q313p!=.

*** watch/clock
g hh_clock= (q313q==1) if q313q!=.

*** bicycle
g hh_bicycle= (q313r==1) if q313r!=.

*** motorcycle or scooter
g hh_motorcycle= (q313s==1) if q313s!=.

*** animal-drawn cart
g hh_animalcart= (q313t==1) if q313t!=.

*** car
g hh_car= (q313u==1) if q313u!=.

*** water pump
g hh_waterpump= (q313v==1) if q313v!=.

*** thresher
g hh_thresher= (q313w==1) if q313w!=.

*** tractor
g hh_tractor= (q313x==1) if q313x!=.

*** livestock- separately for each type?
g hh_cows= (q314a==1) if q314a!=.
g hh_donkey= (q314b==1) if q314b!=.
g hh_goat= (q314c==1) if q314c!=.
g hh_sheep= (q314d==1) if q314d!=.
g hh_chickens= (q314e==1) if q314e!=.

*** electricity
g hh_electricity= (q315==1) if q315!=.

*** rooms used for sleeping
g hh_rooms= q304

*** toilet
tab q311, gen(toilet)

*** Drinking water
tab q310, gen(water)
gen water8 =.
recode water8 (.=0) if q310!=18
recode water8 (.=1) if q310==18
gen water9 =.
recode water9 (.=0) if q310!=96
recode water9 (.=1) if q310==96
// water8 and water9 added, because these categories appear in Round 2 date

*** HH construction
tab q309, gen(hh_construction)

*** Fuel source for cooking
gen fuel_elec = .
	recode fuel_elec (.=1) if q312==1
	recode fuel_elec (.=0) if q312 !=.

gen fuel_lpg = .
	recode fuel_lpg (.=1) if q312==2
	recode fuel_lpg (.=0) if q312 !=.

gen fuel_bio = .
	recode fuel_bio (.=1) if q312==3
	recode fuel_bio (.=0) if q312 !=.

gen fuel_kerosene = .
	recode fuel_kerosene (.=1) if q312==4
	recode fuel_kerosene (.=0) if q312 !=.

gen fuel_coal = .
	recode fuel_coal (.=1) if q312==5 | q312==6
	recode fuel_coal (.=0) if q312 !=.

gen fuel_wood = .
	recode fuel_wood (.=1) if q312==7
	recode fuel_wood (.=0) if q312 !=.

gen fuel_straw = .
	recode fuel_straw (.=1) if q312==8
	recode fuel_straw (.=0) if q312 !=.

gen fuel_crop = .
	recode fuel_crop (.=1) if q312==9
	recode fuel_crop (.=0) if q312 !=.

gen fuel_dung = .
	recode fuel_dung (.=1) if q312==10
	recode fuel_dung (.=0) if q312 !=.
	
*** Internet
g hh_internet= (q316==1) if q316!=.

*** Mosq nets
g hh_nets= (q317==1) if q317!=.

** 7.2 Labelling covariates variables
la var hh_bpl "HAS Yes" 
la var hh_apl "No" 
la var hh_urban "Urban" 
la var hh_rural "Rural"
la var religion_hindu "Hindu"
la var religion_other "Other" 
la var hh_gc "General caste"
la var hh_sc "Scheduled caste" 
la var hh_st "Scheduled tribe" 
la var hh_obc "Other backward caste"
la var mat_ed_none "No education"
la var mat_ed_primary "Some primary"
la var mat_ed_secondary "Some secondary"
la var mat_ed_above "Secondary or above" 
la var parity1 "First birth"
la var parity2 "Second birth"
la var parity3 "Third birth"
la var parity4 "Fourth birth"
la var parity5 "Fifth birth or more"
la var hh_mulbirth "Yes"
la var hh_singbirth "No"
la var qrt_since_birth "Quarters since birth"
la var hh_member "Number of household members"
la var hh_insurance "Household has health insurance"
la var hh_noinsurance "Household has no health insurance"
la var wm_age "eligible woman's age"
la var hh_education "Years of education of the household head"
la var hh_ed_none "Household head education: none"
la var hh_ed_primary "Household head education: some primary"
la var hh_ed_secondary "Household head education: some secondary"
la var hh_ed_above "Household head education: above secondary"
la var hh_educ_group "Household head education group"
la var mat_education "Years of education of the eligible woman"
la var hh_parity "Pregnacy number"
la var hh_mattress "Household owns a mattress"
la var hh_pressurecook "Household owns a pressure cooker" 
la var hh_thermos "Household owns a thremos"
la var hh_chair "Household owns a chair"
la var hh_bed "Household owns a bed"
la var hh_table "Household owns a table"
la var hh_almirah "Household owns a dresser"
la var hh_fan "Household owns a fan"
la var hh_radio "Household owns a radio"
la var hh_tv "Household owns a tv"
la var hh_dvd "Household owns a dvd player"
la var hh_sewingmachine "Household owns a sewing machine"
la var hh_mobile "Household owns a mobile phone"
la var hh_telephone "Household owns a telephone"
la var hh_computer "Household owns a computer"
la var hh_fridge "Household owns a fridge"
la var hh_clock "Household owns a clock"
la var hh_bicycle "Household owns a bicycle"
la var hh_motorcycle "Household owns a motorcycle"
la var hh_animalcart "Household owns an animal cart"
la var hh_car "Household owns a car"
la var hh_waterpump "Household owns a water pump"
la var hh_thresher "Household owns a thresher"
la var hh_tractor "Household owns a tractor"
la var hh_cows "Household owns a cows"
la var hh_donkey "Household owns a donkey"
la var hh_goat "Household owns a goat"
la var hh_sheep "Household owns a sheep"
la var hh_chickens "Household owns a chicken"
la var hh_electricity "Household has electricity"
la var hh_rooms "Number of rooms used for sleeping"
la var water8 "q310==surface water"
la var water9 "q310==other water source"
la var fuel_elec "Household fuel source: electricity"
la var fuel_lpg "Household fuel source: lpg"
la var fuel_bio "Household fuel source: biomass"
la var fuel_kerosene "Household fuel source: kerosene"
la var fuel_coal "Household fuel source: coal"
la var fuel_wood "Household fuel source: wood"
la var fuel_straw "Household fuel source: straw"
la var fuel_crop "Household fuel source: crop"
la var fuel_dung "Household fuel source: dung"
la var hh_internet "Household has internet access"
la var hh_nets "Household has a bednet"

* SECTION 8 - ADDING ADDITIONAL VARIABLE LABELS
la var clustercensus_code "Cluster's census code"
la var triplet "Triplet member"
la var cluster_name "Name of cluster"
la var district "Name of district"
la var district_census_code "District's census code"
la var study_arm "Study arm - intention to treat"
la var sky_actual "Sky provider observed in cluster"
la var urban "Urban cluster"
la var sky_type "Sky provider located in cluster"
la var start_whp_training "Date social franchising training occurred according to WHP"
la var start_selfreport "Date social franchising occurred according to provider"
la var start_branded "Date branding occurred according to provider"
la var birth_m "Month of child's birth"
la var birth_y "Year of child's birth"
la var birth_date "Birth date"
la var birth_date_temp "Birth date"
la var birth_qrt "Quarter of birth"
la var birth_hlf "Semi year of birth"
la var interview_m "Interview month"
la var interview_y "Interview year"
la var aftXintRob1 "Birth occurred after start of franchising and in an intervention cluster where a sky provider was actually observed"
la var aftXintRob3 "Birth occurred after start of franchising and in an intervention cluster"
la var aftXintRob4 "Birth occurred after start of franchising and in an intervention cluster"
la var aftXintRob5 "Birth occurred after start of branding and in an intervention cluster"

* SECTION 9 - SAVING DATA - ONE OBSERVATION PER ELIGIBLE WOMAN (MOST RECENT DELIVERY)
drop if i14!=1
preserve
drop i09- q1127 
save "build\output\master_data.dta", replace

/* SECTION 10 - RESHAPING DATA FOR ALTERNATIVE MORTALITY INDICATORS (ONE OBSERVATION 
				PREGNANCY */ 
restore

** local of pregnancy register variables
local birthVars q514 q515 q516 q517 q519 q520 q522 ///
	q523 q524 q525a q525b q526a q526b q527 q528 q529

drop `birthVars' q521_mm q521_yy birth_date birth_m birth_y birth_date_temp ///
	birth_qrt qrt_since_birth pp3_nmr livebirths pp3_onedaymort

** Renaming variables so that reshape cmnd will work

forvalues i = 1(1)10 {
	foreach x in `birthVars'{ 
		rename `x'_cpy_`i' `x'_`i'
		}  
	rename q521_mm_cpy_`i' q521_mm_`i'
	rename q521_yy_cpy_`i' q521_yy_`i'
}

** Reshaping
reshape long q514_ q515_ q516_ q517_ q519_ q520_ q521_mm_ q521_yy_ q522_ q523_ ///
	q524_ q525a_ q525b_ q526a_ q526b_ q527_ q528_ q529_, i(uid2) j(pregnancy_numb)
	la var pregnancy_num "Order pregnancy recorded in register" 

* SECTION 11 - UPDATING VARIABLES FOR LONG DATA SET 

** 11.1 Birth date and interval between interview date and birth date
g birth_m=q521_mm
g birth_y=q521_yy
g birth_date=ym(birth_y, birth_m)
format birth_date %tm
g birth_date_temp=mdy(birth_m, 1, birth_y)
format birth_date_temp %td
g birth_qrt=qofd(birth_date_temp)
format birth_qrt %tq

la var birth_m "Child's birth month"
la var birth_y "Child's birth year"
la var birth_date "Child's date of birth"

*** Dropping observations where birth_date is missing
drop if birth_date==.
	
*** quarter since birth
gen qrt_since_birth =.
replace  qrt_since_birth = 220 - birth_qrt
la var qrt_since_birth "Quarters since birth"


** 11.2 PP3 outcome variables 
*** pp3_nmr = Neonatal mortality (per 1,000 live births)*full sample of births- see NMR_full sample dofile
gen pp3_nmr=1 if q516==1 & q522==2 & q525a==1 & q525b<=27

replace pp3_nmr=1 if q516==2 & q517==1 & q522==2 & q525a==1 & q525b<=27

*** Not neonatal death if child died on or after 28 days of birth
replace pp3_nmr=0 if q516==1 & q522==2 & q525a==1 & q525b>=28

*** Not neonatal death if child died any number of months after birth
replace pp3_nmr=0 if q516==1 & q522==2 & q525a==2

*** Not neonatal death if child died any number of years after birth
replace pp3_nmr=0 if q516==1 & q522==2 & q525a==3

*** Child born alive and still living
replace pp3_nmr=0 if q516==1 & q522==1

*** Child reported as born dead, showed signs of life at birth, and is still living
replace pp3_nmr=0 if q516==2 & q517==1 & q522==1

*** Child reported as dead, showed signs of life, is now dead, but died 28 or more days after birth
replace pp3_nmr=0 if q516==2 & q517==1 & q522==2 & q525a==1 & q525b>=28

*** Child reported as born dead, showed signs of life at birth, is now dead, died after one month of birth
replace pp3_nmr=0 if q516==2 & q517==1 & q522==2 & q525a==2

*** Child reported as born dead, showed signs of life at birth, is now dead, and died after 1 year of birth
replace pp3_nmr=0 if q516==2 & q517==1 & q522==2 & q525a==3

gen livebirths= (q516==1) if q516!=.

replace livebirths=1 if q516==2 & q517==1 & q516!=.

*** pp3_onedaymort = One-day mortality (per 1,000 live births)
g pp3_onedaymort=1 if q516==1 & q522==2 & q525a==1 & q525b<=1

replace pp3_onedaymort=1 if q516==2 & q517==1 & q522==2 & q525a==1 & q525b<=1

*** Not one day death if child died after 1 days of birth
replace pp3_onedaymort=0 if q516==1 & q522==2 & q525a==1 & q525b>=2

*** Not one day death if child died any number of months after birth
replace pp3_onedaymort=0 if q516==1 & q522==2 & q525a==2

*** Not death if child died any number of years after birth
replace pp3_onedaymort=0 if q516==1 & q522==2 & q525a==3

*** Child born alive and still living
replace pp3_onedaymort=0 if q516==1 & q522==1

*** Child reported as born dead, showed signs of life at birth, and is still living
replace pp3_onedaymort=0 if q516==2 & q517==1 & q522==1

*** Child reported as dead, showed signs of life, is now dead, but died 28 or more days after birth
replace pp3_onedaymort=0 if q516==2 & q517==1 & q522==2 & q525a==1 & q525b>=2

*** Child reported as born dead, showed signs of life at birth, is now dead, died after one month of birth
replace pp3_onedaymort=0 if q516==2 & q517==1 & q522==2 & q525a==2

*** Child reported as born dead, showed signs of life at birth, is now dead, and died after 1 year of birth
replace pp3_onedaymort=0 if q516==2 & q517==1 & q522==2 & q525a==3


* SECTION 12 - SAVING LONG DATA SET (ONE OBSERVATION PER PREGNANCY)

**  Dropping observations before 2015 (so pregrnancies are not double counted from previous survey)
drop if birth_date<tm(2010m1)

**  Dropping unneeded variables
drop i09- q1127

** Saving data
save "build\output\master_data_all_births.dta", replace
