* Do-file to create national level estimates of the burden of neural tube defects

version 14
capture log close
pause on
set more off

*************************************
*Read in .dta file                  *
*************************************
use "C:\NTD_input.dta", clear

* complete NTD prev for countries without data using regional pooled estimate (region2):
gen ov_prev=.
gen ov_prev_lo=.
gen ov_prev_hi=.

* 1. australasia - 
replace ov_prev=12.10 if region2==1
replace ov_prev_lo=10.45 if region2==1
replace ov_prev_hi=13.94 if region2==1

* 2. LAC - 13 datapoints. 3 small islands (DMA, KNA, TTO) and venezuela (VEN) only without mandatory fortification - for other countries use regional meta-analysis
replace ov_prev=7.78 if region2==2 & (iso3!="DMA"| iso3!="KNA"| iso3!="TTO"| iso3!="VEN")
replace ov_prev_lo=6.58 if region2==2 & (iso3!="DMA"| iso3!="KNA"| iso3!="TTO"| iso3!="VEN")
replace ov_prev_hi=8.97 if region2==2 & (iso3!="DMA"| iso3!="KNA"| iso3!="TTO"| iso3!="VEN")

replace ov_prev=22.89 if region2==2 & (iso3=="DMA"| iso3=="KNA"| iso3=="TTO"| iso3=="VEN")
replace ov_prev_lo=18.01 if region2==2 & (iso3=="DMA"| iso3=="KNA"| iso3=="TTO"| iso3=="VEN")
replace ov_prev_hi=28.69 if region2==2 & (iso3=="DMA"| iso3=="KNA"| iso3=="TTO"| iso3=="VEN")

* 3. Eastern europe and central asia. 
replace ov_prev=9.92 if region2==3
replace ov_prev_lo=7.60 if region2==3
replace ov_prev_hi=12.24 if region2==3

replace ov_prev=prev_ntd if region2==3 & use_reported==1
replace ov_prev_lo=low_prev if region2==3 & use_reported==1
replace ov_prev_hi=up_prev if region2==3 & use_reported==1

* 4. sub-Saharan Africa - for countries with >50% coverage of ffa (ZAF, CPV, CMR, CIV, DJI, KEN, RWA, SEN, TZA) tanzania use Sayed post fortification 
replace ov_prev=15.27 if region2==4
replace ov_prev_lo=10.19 if region2==4
replace ov_prev_hi=20.34 if region2==4

replace ov_prev=9.95 if iso3=="ZAF"| iso3=="TZA"| iso3=="CPV"| iso3=="CMR"| iso3=="CIV"| iso3=="DJI"| iso3=="KEN"| iso3=="RWA"| iso3=="SEN"
replace ov_prev_lo=7.26 if iso3=="ZAF"| iso3=="TZA"| iso3=="CPV"| iso3=="CMR"| iso3=="CIV"| iso3=="DJI"| iso3=="KEN"| iso3=="RWA"| iso3=="SEN"
replace ov_prev_hi=13.3 if iso3=="ZAF"| iso3=="TZA"| iso3=="CPV"| iso3=="CMR"| iso3=="CIV"| iso3=="DJI"| iso3=="KEN"| iso3=="RWA"| iso3=="SEN"

* 5. East asia - no fortification. 
replace ov_prev=19.44 if region2==5
replace ov_prev_lo=15.46 if region2==5
replace ov_prev_hi=23.41 if region2==5

* 6. Middle east and north africa 
replace ov_prev=17.45 if region2==6
replace ov_prev_lo=13.56 if region2==6
replace ov_prev_hi=21.34 if region2==6

* 7. Europe - No fortification. use data from countries as is if EUROCAT data available- use regional meta for rest
replace ov_prev=8.63 if region2==7 
replace ov_prev_lo=6.80 if region2==7 
replace ov_prev_hi=10.47 if region2==7

replace ov_prev=prev_ntd if region2==7 & use_reported==1
replace ov_prev_lo=low_prev if region2==7 & use_reported==1
replace ov_prev_hi=up_prev if region2==7 & use_reported==1

* 8. SE Asia - 
replace ov_prev=13.1 if region2==8


* 9. N.America - both countries have data - use own country data
replace ov_prev=prev_ntd if region2==9 & use_reported==1
replace ov_prev_lo=low_prev if region2==9 & use_reported==1
replace ov_prev_hi=up_prev if region2==9 & use_reported==1

* 10. S.asia 
replace ov_prev=31.96 if region2==10
replace ov_prev_lo=23.81 if region2==10
replace ov_prev_hi=40.12 if region2==10

replace ov_prev=9.95 if region2==10 & iso3=="IRN"
replace ov_prev_lo=7.26 if region2==10 & iso3=="IRN"
replace ov_prev_hi=13.3 if region2==10 & iso3=="IRN"


*********************************************************************
*STEP 1 and 2 outputs                                                     *
*********************************************************************
gen n_ntd=(ov_prev*nlb)/10000

*Step 2 - generating split by condition - anenecephaly, encephalocoele, spina bifida
gen com_dat=use_reported
replace com_dat=. if iso3=="RUS"
replace sp_tot=sp_top + sp_sb + sp_lb if sp_tot==. & com_dat==1
replace enc_tot=enc_top + enc_sb + enc_lb if enc_tot==. & com_dat==1
replace an_tot=an_top + an_sb + an_lb if an_tot==. & com_dat==1

gen prop_an=(an_tot/all_tot)
gen prop_enc=(enc_tot/all_tot)
gen prop_sp=(sp_tot/all_tot)

********************************************************************************
* use EUROCAT average for countries without high quality registry data on split*
********************************************************************************
replace prop_an=0.387 if prop_an==.
replace prop_enc=0.121 if prop_enc==.
replace prop_sp=0.492 if prop_sp==.

gen n_an=n_ntd*prop_an
gen n_enc=n_ntd*prop_enc
gen n_sp=n_ntd*prop_sp 

table region2, contents(sum n_ntd sum n_an sum n_enc sum n_sp) format (%9.0fc)

*generate split by outcomes (eTOPFA), stillbirth, livebirth by diagnosis

*ETOPFA
gen prop_an_top=an_top/an_tot
gen prop_sp_top=sp_top/sp_tot
gen prop_enc_top=enc_top/enc_tot

*replacing outcomes if not available in source - based on estimated access to care and PND
*replacing etopfa is not legal in south sudan and ecuador
replace etopfa=0 if etopfa==2| etopfa==4

**********************************************************************************
replace prop_an_top=0.906*prop_access if prop_an_top==. & etopfa>0 
replace prop_sp_top=0.617*prop_access if prop_sp_top==. & etopfa>0 
replace prop_enc_top=0.738*prop_access if prop_enc_top==. & etopfa>0 

replace prop_an_top=0 if prop_an_top==. & etopfa==0 
replace prop_sp_top=0 if prop_sp_top==. & etopfa==0 
replace prop_enc_top=0 if prop_enc_top==. & etopfa==0 


gen n_an_top=n_an*prop_an_top
gen n_sp_top=n_sp*prop_sp_top
gen n_enc_top=n_enc*prop_enc_top

*ntd continuing pregnancies
*an continuing
gen cont_an=n_an-n_an_top
*spb continuing
gen cont_sp=n_sp-n_sp_top
*enc continuing
gen cont_enc=n_enc-n_enc_top


gen prop_an_sb1=0.56
gen prop_an_sb2=0.56
gen prop_an_sb3=0.56
gen prop_sp_sb1=0.031
gen prop_sp_sb2=0.113
gen prop_sp_sb3=0.218
gen prop_enc_sb1=0.036
gen prop_enc_sb2=0.113
gen prop_enc_sb3=0.218

replace etopfa=1 if etopfa>0

gen n_an_sb=(prop_an_sb1*(cont_an*prop_access*etopfa))+(prop_an_sb2*(cont_an*(1-prop_access))) 
replace n_an_sb=(prop_an_sb1*(cont_an*prop_access*etopfa))+(prop_an_sb3*(cont_an*(1-prop_access))) if nmr>=10
*replace n_an_sb=prop_an_rep_sb*cont_an if use_reported==1

gen n_sp_sb=(prop_sp_sb1*(cont_sp*prop_access*etopfa))+(prop_sp_sb2*(cont_sp*(1-prop_access)))
replace n_sp_sb=(prop_sp_sb1*(cont_sp*prop_access*etopfa))+(prop_sp_sb3*(cont_sp*(1-prop_access))) if nmr>=10
*replace n_sp_sb=prop_sp_rep_sb*cont_an if use_reported==1

gen n_enc_sb=(prop_enc_sb1*(cont_enc*prop_access*etopfa))+(prop_enc_sb2*(cont_enc*(1-prop_access))) 
replace n_enc_sb=(prop_enc_sb1*(cont_enc*prop_access*etopfa))+(prop_enc_sb3*(cont_enc*(1-prop_access))) if nmr>=10

*Livebirths
gen n_an_lb=n_an-n_an_top-n_an_sb
gen n_sp_lb=n_sp-n_sp_top-n_sp_sb
gen n_enc_lb=n_enc-n_enc_top-n_enc_sb


gen top_ntd=n_sp_top+n_enc_top+n_an_top
gen sb_ntd=n_sp_sb+n_enc_sb+n_an_sb
gen lb_ntd=n_sp_lb+n_enc_lb+n_an_lb


*overall table
table region2, contents(sum n_ntd sum top_ntd sum sb_ntd sum lb_ntd) format (%9.0fc)
*spina bifida table
table region2, contents(sum n_sp sum n_sp_top sum n_sp_sb sum n_sp_lb) format (%9.0fc)
*encephalocoele table
table region2, contents(sum n_enc sum n_enc_top sum n_enc_sb sum n_enc_lb) format (%9.0fc)
*anencephaly table
table region2, contents(sum n_an sum n_an_top sum n_an_sb sum n_an_lb) format (%9.0fc)

graph bar (sum)n_ntd (sum) top_ntd (sum)sb_ntd (sum) lb_ntd , over(region2, sort(n_ntd) label(angle(forty_five) labsize(small))) ytitle("# pregnancies affected by NTDs") ytitle(,size(small) orientation(vertical)) ylabel(, labels labsize(small) angle(horizontal)format(%6.0fc)valuelabel nogrid) legend(on) legend(order(1 "Total NTDs" 2 "NTDs - eTOPFA" 3 "NTDs - stillbirths" 4 "NTDs - livebirths"))

****************************************************
*section to create regional birth prevalence rates *
****************************************************
capture drop count
replace country=whoname if country==""

*Overall NTD birth prevalence:
   encode country,gen(idcode)
   gen ave_ntdr=n_ntd/nlb 
   egen ntd_reg=sum(n_ntd),by(region2)
   egen birthsreg=sum(nlb),by(region2)
   gen ntdr_reg=ntd_reg*10000/birthsreg
   sort region2
   by region2: gen count=_n
   table region2, c (mean ntdr_reg)
 
   summ ave_ntdr,detail
   quietly {
      summ nlb
	  local lbtot=r(sum)
	  summ n_ntd
	  local ntd=r(sum)
	  local ntdr= `ntd'*10000/(`lbtot')
      noisily di "Global NTDR = ",`ntdr'
   }

*

*Anencephaly birth prevalence
capture drop count
   gen ave_anr=n_an/nlb 
   egen an_reg=sum(n_an),by(region2)
   gen anr_reg=an_reg*10000/birthsreg
   sort region2
   by region2: gen count=_n
   table region2, c (mean anr_reg)
 
   summ ave_anr,detail
   
   noisily {
      summ nlb
	  local lbtot=r(sum)
	  summ n_an
	  local an=r(sum)
	  local anr= `an'*10000/(`lbtot')
      noisily di "Global Anencephaly Ratio = ",`anr'
   }

*
* Spina Bifida birth prevalence
capture drop count
   gen ave_spr=n_sp/nlb 
   egen sp_reg=sum(n_sp),by(region2)
   gen spr_reg=sp_reg*10000/birthsreg
   sort region2
   by region2: gen count=_n
   table region2, c (mean spr_reg)
 
   summ ave_spr,detail
  
   noisily {
      summ nlb
	  local lbtot=r(sum)
	  summ n_sp
	  local sp=r(sum)
	  local spr= `sp'*10000/(`lbtot')
      noisily di "Global Spina Bifida Ratio = ",`spr'
   }

*
* Encephalocoele birth prevalence
capture drop count
   gen ave_encr=n_enc/nlb 
   egen enc_reg=sum(n_enc),by(region2)
   gen encr_reg=enc_reg*10000/birthsreg
   sort region2
   by region2: gen count=_n
   table region2, c (mean encr_reg)
 
   summ ave_encr,detail
 
   noisily {
      summ nlb
	  local lbtot=r(sum)
	  summ n_enc
	  local enc=r(sum)
	  local encr= `enc'*10000/(`lbtot')
      noisily di "Global Encephalocoele Ratio = ",`encr'
   }

*

****************************************************************************
*creating summaries of measure by outcome - eTOPFA, stillbirth, livebirths *
****************************************************************************
*eTOPFA for NTD regional and global prevalence
capture drop count
   gen ave_topr=top_ntd/nlb 
   egen top_reg=sum(top_ntd),by(region2)
   gen topr_reg=top_reg*10000/birthsreg
   sort region2
   by region2: gen count=_n
   table region2, c (mean topr_reg)
 
   summ ave_topr,detail
  
   noisily {
      summ nlb
	  local lbtot=r(sum)
	  summ top_ntd
	  local top=r(sum)
	  local topr= `top'*10000/(`lbtot')
      noisily di "Global NTD eTOPFA Ratio = ",`topr'
   }

*
*stillbirths from NTD regional and global prevalence
capture drop count
   gen ave_sbr=sb_ntd/nlb 
   egen sb_reg=sum(sb_ntd),by(region2)
   gen sbr_reg=sb_reg*10000/birthsreg
   sort region2
   by region2: gen count=_n
   table region2, c (mean sbr_reg)
 
   summ ave_sbr,detail
  
   noisily {
      summ nlb
	  local lbtot=r(sum)
	  summ sb_ntd
	  local sb=r(sum)
	  local sbr= `sb'*10000/(`lbtot')
      noisily di "Global NTD stillbirth Ratio = ",`sbr'
   }

*
*livebirths with NTD regional and global prevalence
capture drop count
   gen ave_lbr=lb_ntd/nlb 
   egen lb_reg=sum(lb_ntd),by(region2)
   gen lbr_reg=lb_reg*10000/birthsreg
   sort region2
   by region2: gen count=_n
   table region2, c (mean lbr_reg)
 
   summ ave_lbr,detail
  
   noisily {
      summ nlb
	  local lbtot=r(sum)
	  summ lb_ntd
	  local lb=r(sum)
	  local lbr= `lb'*10000/(`lbtot')
      noisily di "Global NTD livebirth Ratio = ",`lbr'
   }

*
save "C:\ntd_output.dta", replace

***************************************************************************************
*Creating uncertainty around reported rates
***************************************************************************************

replace prop_an=. if use_reported!=1
replace prop_sp=. if use_reported!=1
replace prop_enc=. if use_reported!=1

gen prop_an_rep=prop_an
gen prop_sp_rep=prop_sp
gen prop_enc_rep=prop_enc

replace prop_an_top=. if use_reported!=1
replace prop_sp_top=. if use_reported!=1
replace prop_enc_top=. if use_reported!=1

gen prop_an_top_rep=prop_an_top
gen prop_sp_top_rep=prop_sp_top
gen prop_enc_top_rep=prop_enc_top

replace prop_an_sb=. if use_reported!=1
replace prop_sp_sb=. if use_reported!=1
replace prop_enc_sb=. if use_reported!=1

gen prop_an_rep_sb=prop_an_sb
gen prop_sp_rep_sb=prop_sp_sb
gen prop_enc_rep_sb=prop_enc_sb

keep iso3 pop_lb nlb prev_ntd use_reported region region2 prop_access u5mr nmr imr etopfa prop_an_rep prop_enc_rep prop_sp_rep prop_an_top_rep prop_enc_top_rep prop_sp_top_rep prop_an_rep_sb prop_enc_rep_sb prop_sp_rep_sb

save "C:\reported_rates.dta", replace

local i=1
local numsim=1000
quietly while `i'<=1000 {
    use "C:\reported_rates.dta", clear
	gen n_ntd=(prev_ntd*pop_lb)/10000
    gen n_sample=`i'
    gen se=sqrt(n_ntd)
    gen uncertain=round(n_ntd+se*rnormal())
    replace n_ntd=max(0,uncertain)
    gen prev_ntd_report=(n_ntd/pop_lb)*10000
    if `i'>1 {
       append using "C:\RR_range.dta"
    }
    save "C:\RR_range.dta", replace
 
    local i=`i'+1
 }

*

use "C:\RR_range.dta", clear
replace n_ntd=. 
gen sim=n_sample
drop n_sample
gen l_etopfa=.
replace l_etopfa=1 if etopfa>0
replace l_etopfa=0 if etopfa==0

sort sim

save "C:\RR_range.dta", replace

*******************************************************************************************
*SECTION TO CREATE UNCERTAINTY
*******************************************************************************************
clear
set seed 65427
set obs 1000

*********************************************************
*           Step 1 overall NTD prevalence               *
*********************************************************

* 1. australasia 
gen ntd_reg1=rpoisson(12.1) 

* 2. LAC -
gen ntd_reg2_ffa=rpoisson(7.78)
gen ntd_reg2=rpoisson(22.89)

* 3. Eastern europe and central asia. Use country own data for countries with umbrella registry data. meta for countries without effective fortification. For countries with fortification - at acceptable levels and coverage (>200mg/day and >70% coverage  - only turkmenistan (95% reduction in folate preventable NTDs) and uzbekistan (80% reduction in folate preventable NTDs)
gen ntd_reg3=rpoisson(9.92)

* 4. sub-Saharan Africa - for countries with >50% coverage of ffa (ZAF, CPV, CMR, CIV, DJI, KEN, RWA, SEN, TZA) tanzania use Sayed post fortification 
gen ntd_reg4=rpoisson(15.27)
gen ntd_reg4_ffa=rpoisson(9.95)

* 5. East asia - no fortification. 
gen ntd_reg5=rpoisson(19.44)

* 6. Middle east and north africa 
gen ntd_reg6=rpoisson(17.45)

* 7. Europe - No fortification. use data from countries as is if EUROCAT data available- use regional meta for rest
gen ntd_reg7=rpoisson(8.63)

* 8. SE Asia -  
gen ntd_reg8=rpoisson(13.1)

* 9. N.America - both countries have data - use own country data

* 10. S.asia - iran - only country with high level of mandatory fortification - use own data. For all other countries use regional meta 
gen ntd_reg10=rpoisson(31.96)

*********************************************************************************************************************
*Step 2 - generating split by condition - anenecephaly, encephalocoele, spina bifida
*********************************************************************************************************************
gen prop_an=rnormal(0.387,0.056)
gen prop_spenc=1-prop_an
gen prop_sp=0.80*prop_spenc
gen prop_enc=0.2*prop_spenc

*********************************************************************************************************************
*Step 3 - generating split by birth outcome
*********************************************************************************************************************

*generate split by outcomes (eTOPFA), stillbirth, livebirth by diagnosis
*an eTOPFA
gen prop_an_top=rnormal(0.906,0.012)
gen prop_sp_top=rnormal(0.617, 0.031)
gen prop_enc_top=rnormal(0.738,0.027)

*stillbirths in continuing pregnancies
gen prop_an_sb1=rnormal(0.56,0.16)
gen prop_an_sb2=prop_an_sb1
gen prop_an_sb3=prop_an_sb1

gen prop_sp_sb1=rnormal(0.031,0.013)
gen prop_sp_sb2=rnormal(0.113,0.031)
gen prop_sp_sb3=rnormal(0.218,0.022)

gen prop_enc_sb1=rnormal(0.036,0.009)
gen prop_enc_sb2=rnormal(0.113,0.031)
gen prop_enc_sb3=rnormal(0.218,0.022)


*********************************************************
*section 4 under 5 case fatality                       *
*********************************************************
gen anCFR=1

gen spCFR_nocare=rnormal(0.90,0.051)
gen spCFR_care2=rnormal(0.183,0.035)
gen spCFR_care3=rnormal(0.467,0.043)

gen encCFR_nocare=rnormal(0.90,0.051)
gen encCFR_care2=rnormal(0.279,0.020)
gen encCFR_care3=rnormal(0.467,0.043)

gen count=_n
gen sim=count
drop count

save "C:\ntd_uncertainty",replace 

*********************************************************
*section to merge required data files                   *
*********************************************************

** To merge the data with the prevalence (and other) data by country for countries with data
*need to create 1000 draws of poisson (using code from stillbirths reported) for those who we are using data as reported (24 countries)
sort sim
merge sim using "C:\RR_range.dta"

***************************************************************************************************************************
* Section on overall NTD envelop uncertainty
**************************************************************************************
*generating number of NTD overall by region for 1000 sims:
*gen n_ntd=.
replace n_ntd=(ntd_reg1*nlb)/10000 if region2==1
replace n_ntd=(ntd_reg2*nlb)/10000 if region2==2
replace n_ntd=(ntd_reg2_ffa*nlb)/10000 if region2==2 & (iso3!="DMA"| iso3!="KNA"| iso3!="TTO"| iso3!="VEN")
replace n_ntd=(ntd_reg3*nlb)/10000 if region2==3
replace n_ntd=(ntd_reg4*nlb)/10000 if region2==4
replace n_ntd=(ntd_reg5*nlb)/10000 if region2==5
replace n_ntd=(ntd_reg6*nlb)/10000 if region2==6
replace n_ntd=(ntd_reg7*nlb)/10000 if region2==7
replace n_ntd=(ntd_reg8*nlb)/10000 if region2==8
replace n_ntd=(ntd_reg10*nlb)/10000 if region2==10

* Replacing reported rates with uncertainty where available (24 countries)
replace n_ntd=(prev_ntd_report*nlb)/10000 if prev_ntd_report!=.

* Regional uncertainty NTD prevalence overall 
preserve
	collapse (sum) n_ntd, by(sim region2)
	save dta\ntd_uncertainty_region_range,replace
	sort region2 n_ntd
	keep region2 n_ntd
	by region2: gen count=_n
	keep if count==25 | count==975
	gen low_ntd=n_ntd if count==25
	gen hi_ntd=n_ntd if count==975
	collapse (max) low_ntd hi_ntd, by(region2)
	save dta\ntd_regionaluncertainty,replace
restore

*global uncertainty NTD prevalence overall
preserve
collapse (sum) n_ntd, by(sim)
  centile n_ntd, centile(2.5 97.5)
  save dta\ntd_uncertainty_global_range,replace
restore


**************************************************************************
*uncertainty around type split
**************************************************************************
*for countries with reported rates for now assuming no uncertainty in the split
gen n_an=n_ntd*prop_an
gen n_sp=n_ntd*prop_sp
gen n_enc=n_ntd*prop_enc

replace n_an=n_ntd*prop_an_rep if use_reported==1
replace n_sp=n_ntd*prop_sp_rep if use_reported==1
replace n_enc=n_ntd*prop_enc_rep if use_reported==1

* Regional uncertainty NTD subtype numbers 
preserve
	collapse (sum) n_an, by(sim region2)
	save dta\an_uncertainty_region_range,replace
	sort region2 n_an
	keep region2 n_an
	by region2: gen count=_n
	keep if count==25 | count==975
	gen low_an=n_an if count==25
	gen hi_an=n_an if count==975
	collapse (max) low_an hi_an, by(region2)
	save dta\an_regionaluncertainty,replace
restore

preserve
	collapse (sum) n_sp, by(sim region2)
	save dta\sp_uncertainty_region_range,replace
	sort region2 n_sp
	keep region2 n_sp
	by region2: gen count=_n
	keep if count==25 | count==975
	gen low_sp=n_sp if count==25
	gen hi_sp=n_sp if count==975
	collapse (max) low_sp hi_sp, by(region2)
	save dta\sp_regionaluncertainty,replace
restore

preserve
	collapse (sum) n_enc, by(sim region2)
	save dta\enc_uncertainty_region_range,replace
	sort region2 n_enc
	keep region2 n_enc
	by region2: gen count=_n
	keep if count==25 | count==975
	gen low_enc=n_enc if count==25
	gen hi_enc=n_enc if count==975
	collapse (max) low_enc hi_enc, by(region2)
	save dta\enc_regionaluncertainty,replace
restore

*global uncertainty NTD prevalence subtype
preserve
collapse (sum) n_an, by(sim)
  centile n_an, centile(2.5 97.5)
  save dta\an_uncertainty_global_range,replace
restore

preserve
collapse (sum) n_sp, by(sim)
  centile n_sp, centile(2.5 97.5)
  save dta\sp_uncertainty_global_range,replace
restore

preserve
collapse (sum) n_enc, by(sim)
  centile n_enc, centile(2.5 97.5)
  save dta\enc_uncertainty_global_range,replace
restore


**************************************************************************
*uncertainty around proportion eTOPFA and number of continuing pregnancies
****************************************************************************
*number of eTOPFA - anencephaly
gen n_an_top=n_an*prop_an_top*prop_access*l_etopfa

replace n_an_top=n_an*prop_an_top_rep if use_reported==1

*number of eTOPFA - spina bifida
gen n_sp_top=n_sp*prop_sp_top*prop_access*l_etopfa

replace n_sp_top=n_sp*prop_sp_top_rep if use_reported==1

*number of eTOPFA - encephalocoele
gen n_enc_top=n_enc*prop_enc_top*prop_access*l_etopfa

replace n_enc_top=n_enc*prop_enc_top_rep if use_reported==1

*total number of eTOPFA for NTDS
gen n_tot_top=n_an_top+n_sp_top+n_enc_top

* Regional uncertainty NTD eTOPFA subtype numbers 
*total
preserve
	collapse (sum) n_tot_top, by(sim region2)
	save dta\tot_top_uncertainty_region_range,replace
	sort region2 n_tot_top
	keep region2 n_tot_top
	by region2: gen count=_n
	keep if count==25 | count==975
	gen low_tot_top=n_tot_top if count==25
	gen hi_tot_top=n_tot_top if count==975
	collapse (max) low_tot_top hi_tot_top, by(region2)
	save dta\tot_top_regionaluncertainty,replace
restore
*anencephaly
preserve
	collapse (sum) n_an_top, by(sim region2)
	save dta\an_top_uncertainty_region_range,replace
	sort region2 n_an_top
	keep region2 n_an_top
	by region2: gen count=_n
	keep if count==25 | count==975
	gen low_an_top=n_an_top if count==25
	gen hi_an_top=n_an_top if count==975
	collapse (max) low_an_top hi_an_top, by(region2)
	save dta\an_top_regionaluncertainty,replace
restore
*spina bifida - top
preserve
	collapse (sum) n_sp_top, by(sim region2)
	save dta\sp_top_uncertainty_region_range,replace
	sort region2 n_sp_top
	keep region2 n_sp_top
	by region2: gen count=_n
	keep if count==25 | count==975
	gen low_sp_top=n_sp_top if count==25
	gen hi_sp_top=n_sp_top if count==975
	collapse (max) low_sp_top hi_sp_top, by(region2)
	save dta\sp_top_regionaluncertainty,replace
restore
*enc -top
preserve
	collapse (sum) n_enc_top, by(sim region2)
	save dta\enc_top_uncertainty_region_range,replace
	sort region2 n_enc_top
	keep region2 n_enc_top
	by region2: gen count=_n
	keep if count==25 | count==975
	gen low_enc_top=n_enc_top if count==25
	gen hi_enc_top=n_enc_top if count==975
	collapse (max) low_enc_top hi_enc_top, by(region2)
	save dta\enc_top_regionaluncertainty,replace
restore

*global uncertainty NTD eTOPFA and by subtype
preserve
collapse (sum) n_tot_top, by(sim)
  centile n_tot_top, centile(2.5 97.5)
  save dta\tot_top_uncertainty_global_range,replace
restore

preserve
collapse (sum) n_an_top, by(sim)
  centile n_an_top, centile(2.5 97.5)
  save dta\an_top_uncertainty_global_range,replace
restore

preserve
collapse (sum) n_sp_top, by(sim)
  centile n_sp_top, centile(2.5 97.5)
  save dta\sp_top_uncertainty_global_range,replace
restore

preserve
collapse (sum) n_enc_top, by(sim)
  centile n_enc_top, centile(2.5 97.5)
  save dta\enc_top_uncertainty_global_range,replace
restore


*ntd continuing pregnancies
*an continuing
gen cont_an=n_an-n_an_top
*spb continuing
gen cont_sp=n_sp-n_sp_top
*enc continuing
gen cont_enc=n_enc-n_enc_top

*generating number of stillbirths
*gen n_sp_sb=(number of stillbirths in ongoing pregnancies with access to PND and TOP care)+(number of stillbirths in ongoing pregnancies with no access to PND and TOP care)
gen n_an_sb=(prop_an_sb1*(cont_an*prop_access*l_etopfa))+(prop_an_sb2*(cont_an*(1-prop_access))) 
replace n_an_sb=(prop_an_sb1*(cont_an*prop_access*l_etopfa))+(prop_an_sb3*(cont_an*(1-prop_access))) if nmr>=10
*replace n_an_sb=prop_an_rep_sb*cont_an if use_reported==1

gen n_sp_sb=(prop_sp_sb1*(cont_sp*prop_access*l_etopfa))+(prop_sp_sb2*(cont_sp*(1-prop_access)))
replace n_sp_sb=(prop_sp_sb1*(cont_sp*prop_access*l_etopfa))+(prop_sp_sb3*(cont_sp*(1-prop_access))) if nmr>=10
*replace n_sp_sb=prop_sp_rep_sb*cont_an if use_reported==1

gen n_enc_sb=(prop_enc_sb1*(cont_enc*prop_access*l_etopfa))+(prop_enc_sb2*(cont_enc*(1-prop_access))) 
replace n_enc_sb=(prop_enc_sb1*(cont_enc*prop_access*l_etopfa))+(prop_enc_sb3*(cont_enc*(1-prop_access))) if nmr>=10
*replace n_enc_sb=prop_enc_rep_sb*cont_an if use_reported==1

*total number of stillbirths associated with NTDS
gen n_tot_sb=n_an_sb+n_sp_sb+n_enc_sb 

* Regional uncertainty number of ntd stillbirths
*total
preserve
	collapse (sum) n_tot_sb, by(sim region2)
	save dta\tot_sb_uncertainty_region_range,replace
	sort region2 n_tot_sb
	keep region2 n_tot_sb
	by region2: gen count=_n
	keep if count==25 | count==975
	gen low_tot_sb=n_tot_sb if count==25
	gen hi_tot_sb=n_tot_sb if count==975
	collapse (max) low_tot_sb hi_tot_sb, by(region2)
	save dta\tot_sb_regionaluncertainty,replace
restore
*anencephaly
preserve
	collapse (sum) n_an_sb, by(sim region2)
	save dta\an_sb_uncertainty_region_range,replace
	sort region2 n_an_sb
	keep region2 n_an_sb
	by region2: gen count=_n
	keep if count==25 | count==975
	gen low_an_sb=n_an_sb if count==25
	gen hi_an_sb=n_an_sb if count==975
	collapse (max) low_an_sb hi_an_sb, by(region2)
	save dta\an_sb_regionaluncertainty,replace
restore
*spina bifida - sb
preserve
	collapse (sum) n_sp_sb, by(sim region2)
	save dta\sp_sb_uncertainty_region_range,replace
	sort region2 n_sp_sb
	keep region2 n_sp_sb
	by region2: gen count=_n
	keep if count==25 | count==975
	gen low_sp_sb=n_sp_sb if count==25
	gen hi_sp_sb=n_sp_sb if count==975
	collapse (max) low_sp_sb hi_sp_sb, by(region2)
	save dta\sp_sb_regionaluncertainty,replace
restore
*enc -sb
preserve
	collapse (sum) n_enc_sb, by(sim region2)
	save dta\enc_sb_uncertainty_region_range,replace
	sort region2 n_enc_sb
	keep region2 n_enc_sb
	by region2: gen count=_n
	keep if count==25 | count==975
	gen low_enc_sb=n_enc_sb if count==25
	gen hi_enc_sb=n_enc_sb if count==975
	collapse (max) low_enc_sb hi_enc_sb, by(region2)
	save dta\enc_sb_regionaluncertainty,replace
restore

*global uncertainty NTD stillbirths and by subtype
preserve
collapse (sum) n_tot_sb, by(sim)
  centile n_tot_sb, centile(2.5 97.5)
  save dta\tot_sb_uncertainty_global_range,replace
restore

preserve
collapse (sum) n_an_sb, by(sim)
  centile n_an_sb, centile(2.5 97.5)
  save dta\an_sb_uncertainty_global_range,replace
restore

preserve
collapse (sum) n_sp_sb, by(sim)
  centile n_sp_sb, centile(2.5 97.5)
  save dta\sp_sb_uncertainty_global_range,replace
restore

preserve
collapse (sum) n_enc_sb, by(sim)
  centile n_enc_sb, centile(2.5 97.5)
  save dta\enc_sb_uncertainty_global_range,replace
restore

pause
************************************************
*generating number of affected livebirths
gen n_an_lb=n_an-n_an_top-n_an_sb
gen n_sp_lb=n_sp-n_sp_top-n_sp_sb
gen n_enc_lb=n_enc-n_enc_top-n_enc_sb

gen top_ntd=n_sp_top+n_enc_top+n_an_top
gen sb_ntd=n_sp_sb+n_enc_sb+n_an_sb
gen lb_ntd=n_sp_lb+n_enc_lb+n_an_lb


*total number of livebirths associated with NTDS
gen n_tot_lb=n_an_lb+n_sp_lb+n_enc_lb 

* Regional uncertainty number of ntd livebirths
*total
preserve
	collapse (sum) n_tot_lb, by(sim region2)
	save dta\tot_lb_uncertainty_region_range,replace
	sort region2 n_tot_lb
	keep region2 n_tot_lb
	by region2: gen count=_n
	keep if count==25 | count==975
	gen low_tot_lb=n_tot_lb if count==25
	gen hi_tot_lb=n_tot_lb if count==975
	collapse (max) low_tot_lb hi_tot_lb, by(region2)
	save dta\tot_lb_regionaluncertainty,replace
restore
*anencephaly
preserve
	collapse (sum) n_an_lb, by(sim region2)
	save dta\an_lb_uncertainty_region_range,replace
	sort region2 n_an_lb
	keep region2 n_an_lb
	by region2: gen count=_n
	keep if count==25 | count==975
	gen low_an_lb=n_an_lb if count==25
	gen hi_an_lb=n_an_lb if count==975
	collapse (max) low_an_lb hi_an_lb, by(region2)
	save dta\an_lb_regionaluncertainty,replace
restore
*spina bifida 
preserve
	collapse (sum) n_sp_lb, by(sim region2)
	save dta\sp_lb_uncertainty_region_range,replace
	sort region2 n_sp_lb
	keep region2 n_sp_lb
	by region2: gen count=_n
	keep if count==25 | count==975
	gen low_sp_lb=n_sp_lb if count==25
	gen hi_sp_lb=n_sp_lb if count==975
	collapse (max) low_sp_lb hi_sp_lb, by(region2)
	save dta\sp_lb_regionaluncertainty,replace
restore
*encephalocoele 
preserve
	collapse (sum) n_enc_lb, by(sim region2)
	save dta\enc_lb_uncertainty_region_range,replace
	sort region2 n_enc_lb
	keep region2 n_enc_lb
	by region2: gen count=_n
	keep if count==25 | count==975
	gen low_enc_lb=n_enc_lb if count==25
	gen hi_enc_lb=n_enc_lb if count==975
	collapse (max) low_enc_lb hi_enc_lb, by(region2)
	save dta\enc_bb_regionaluncertainty,replace
restore

*global uncertainty NTD eTOPFA and by subtype
preserve
collapse (sum) n_tot_lb, by(sim)
  centile n_tot_lb, centile(2.5 97.5)
  save dta\tot_lb_uncertainty_global_range,replace
restore

preserve
collapse (sum) n_an_lb, by(sim)
  centile n_an_lb, centile(2.5 97.5)
  save dta\an_lb_uncertainty_global_range,replace
restore

preserve
collapse (sum) n_sp_lb, by(sim)
  centile n_sp_lb, centile(2.5 97.5)
  save dta\sp_lb_uncertainty_global_range,replace
restore

preserve
collapse (sum) n_enc_lb, by(sim)
  centile n_enc_lb, centile(2.5 97.5)
  save dta\enc_lb_uncertainty_global_range,replace
restore

pause
******************************
*Uncertainty around under 5 deaths
*****************************
gen an_u5d=n_an_lb*anCFR

gen sp_u5d=.
replace sp_u5d=(n_sp_lb*spCFR_nocare*(1-prop_access))+ (n_sp_lb*spCFR_care2*prop_access) if nmr<10
replace sp_u5d=(n_sp_lb*spCFR_nocare*(1-prop_access))+ (n_sp_lb*spCFR_care3*prop_access) if nmr>=10

gen enc_u5d=.
replace enc_u5d=(n_enc_lb*encCFR_nocare*(1-prop_access))+ (n_enc_lb*encCFR_care2*prop_access) if nmr<10
replace enc_u5d=(n_enc_lb*encCFR_nocare*(1-prop_access))+ (n_enc_lb*encCFR_care3*prop_access) if nmr>=10

gen u5d_ntd=an_u5d+sp_u5d+enc_u5d

* Regional uncertainty overall ntd -NND 
preserve
	collapse (sum) u5d_ntd, by(sim region2)
	save dta\u5d_ntd_uncertainty_region_range,replace
	sort region2 u5d_ntd
	keep region2 u5d_ntd
	by region2: gen count=_n
	keep if count==25 | count==975
	gen lowu5d=u5d_ntd if count==25
	gen hiu5d=u5d_ntd if count==975
	collapse (max) lowu5d hiu5d, by(region2)
	save dta\u5d_ntd_regionaluncertainty,replace
restore

*global uncertainty overall ntd-NND
preserve
collapse (sum) u5d_ntd, by(sim)
  centile u5d_ntd, centile(2.5 97.5)
  save dta\u5d_uncertainty_global_range,replace
restore

*******************
* Regional uncertainty anencephaly -NND 
preserve
	collapse (sum) an_u5d, by(sim region2)
	save dta\u5d_an_uncertainty_region_range,replace
	sort region2 an_u5d
	keep region2 an_u5d
	by region2: gen count=_n
	keep if count==25 | count==975
	gen low_anu5d=an_u5d if count==25
	gen hi_anu5d=an_u5d if count==975
	collapse (max) low_anu5d hi_anu5d, by(region2)
	save dta\u5d_an_regionaluncertainty,replace
restore

*global uncertainty anencephaly-NND
preserve
collapse (sum) an_u5d, by(sim)
  centile an_u5d, centile(2.5 97.5)
  save dta\an_u5d_uncertainty_global_range,replace
restore

* Regional uncertainty spina bifida -NND 
preserve
	collapse (sum) sp_u5d, by(sim region2)
	save dta\u5d_sp_uncertainty_region_range,replace
	sort region2 sp_u5d
	keep region2 sp_u5d
	by region2: gen count=_n
	keep if count==25 | count==975
	gen low_spu5d=sp_u5d if count==25
	gen hi_spu5d=sp_u5d if count==975
	collapse (max) low_spu5d hi_spu5d, by(region2)
	save dta\u5d_sp_regionaluncertainty,replace
restore

*global uncertainty spina bifida-NND
preserve
collapse (sum) sp_u5d, by(sim)
  centile sp_u5d, centile(2.5 97.5)
  save dta\sp_u5d_uncertainty_global_range,replace
restore

* Regional uncertainty encephalocoele -NND 
preserve
	collapse (sum) enc_u5d, by(sim region2)
	save dta\u5d_enc_uncertainty_region_range,replace
	sort region2 enc_u5d
	keep region2 enc_u5d
	by region2: gen count=_n
	keep if count==25 | count==975
	gen low_encu5d=enc_u5d if count==25
	gen hi_encu5d=enc_u5d if count==975
	collapse (max) low_encu5d hi_encu5d, by(region2)
	save dta\u5d_enc_regionaluncertainty,replace
restore

*global uncertainty encephalocoele-NND
preserve
collapse (sum) enc_u5d, by(sim)
  centile enc_u5d, centile(2.5 97.5)
  save dta\enc_u5d_uncertainty_global_range,replace
restore

pause
******************************************************************************
*section on regional breakdown of number of livebirths                       *
******************************************************************************

* Regional livebirths
preserve
    keep if sim==1
	collapse (sum) nlb, by(sim region2)
	save dta\regional_nlb, replace
	collapse (max) nlb, by(sim region2)
	save dta\regional_nlb2, replace
restore


