/*
 * Decompiled with CFR 0.152.
 */
package com.machinezoo.sourceafis;

import com.machinezoo.sourceafis.FingerprintTemplate;
import com.machinezoo.sourceafis.ForeignDimensions;
import com.machinezoo.sourceafis.ForeignFingerprint;
import com.machinezoo.sourceafis.ForeignFormat;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.EOFException;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class ForeignTemplate {
    private static final Logger logger = LoggerFactory.getLogger(ForeignTemplate.class);
    ForeignFormat format;
    ForeignDimensions dimensions;
    List<ForeignFingerprint> fingerprints = new ArrayList<ForeignFingerprint>();

    ForeignTemplate(FingerprintTemplate ... templates) {
        for (FingerprintTemplate template : templates) {
            this.fingerprints.add(new ForeignFingerprint(template));
        }
        this.dimensions = this.fingerprints.stream().findFirst().map(f -> f.dimensions).orElse(null);
    }

    ForeignTemplate(DataInputStream in) throws IOException {
        this.readFormatMarker(in);
        this.readVersion(in);
        this.readTemplateLength(in);
        if (this.format == ForeignFormat.ISO_19794_2_2005) {
            in.skipBytes(2);
        }
        if (this.format != ForeignFormat.ISO_19794_2_2005) {
            this.readProductId(in);
            this.readSensorInfo(in);
        }
        if (this.format == ForeignFormat.ANSI_378_2004 || this.format == ForeignFormat.ISO_19794_2_2005) {
            this.dimensions = new ForeignDimensions(in);
        }
        int count = this.readFingerprintCount(in);
        in.skipBytes(1);
        for (int i = 0; i < count; ++i) {
            this.fingerprints.add(new ForeignFingerprint(in, this.format, this.dimensions));
        }
        if (this.format == ForeignFormat.ANSI_378_2009 || this.format == ForeignFormat.ANSI_378_2009_AM1) {
            this.dimensions = this.fingerprints.stream().findFirst().map(f -> f.dimensions).orElse(null);
        }
    }

    static ForeignTemplate read(byte[] template) {
        try {
            return new ForeignTemplate(new DataInputStream(new ByteArrayInputStream(template)));
        }
        catch (EOFException ex) {
            throw new IllegalArgumentException("Unexpected end of template data");
        }
        catch (IOException ex) {
            throw new IllegalStateException("Unexpected I/O error", ex);
        }
    }

    void write(DataOutputStream out) throws IOException {
        if (this.format == ForeignFormat.ISO_19794_2_2005) {
            throw new IllegalStateException();
        }
        this.writeFormatMarker(out);
        this.writeVersion(out);
        this.writeTemplateLength(out);
        this.writeProductId(out);
        this.writeSensorInfo(out);
        if (this.format == ForeignFormat.ANSI_378_2004) {
            this.dimensions.write(out);
        }
        this.writeFingerprintCount(out);
        out.writeByte(0);
        for (int i = 0; i < this.fingerprints.size(); ++i) {
            this.fingerprints.get(i).write(out, this.format, i);
        }
    }

    byte[] write() {
        try {
            ByteArrayOutputStream buffer = new ByteArrayOutputStream();
            this.write(new DataOutputStream(buffer));
            return buffer.toByteArray();
        }
        catch (IOException ex) {
            throw new IllegalStateException("Unexpected I/O error", ex);
        }
    }

    private void readFormatMarker(DataInputStream in) throws IOException {
        if (in.readByte() != 70 || in.readByte() != 77 || in.readByte() != 82 || in.readByte() != 0) {
            throw new IllegalArgumentException("Unsupported template format: missing FMR signature at the beginning");
        }
    }

    private void writeFormatMarker(DataOutputStream out) throws IOException {
        out.writeByte(70);
        out.writeByte(77);
        out.writeByte(82);
        out.writeByte(0);
    }

    private void readVersion(DataInputStream in) throws IOException {
        byte v0 = in.readByte();
        byte v1 = in.readByte();
        byte v2 = in.readByte();
        byte v3 = in.readByte();
        if (v0 == 32 && v1 == 50 && v2 == 48 && v3 == 0) {
            this.format = ForeignFormat.ANSI_378_2004;
        } else if (v0 == 48 && v1 == 51 && v2 == 48 && v3 == 0) {
            this.format = ForeignFormat.ANSI_378_2009;
        } else if (v0 == 48 && v1 == 51 && v2 == 53 && v3 == 0) {
            this.format = ForeignFormat.ANSI_378_2009_AM1;
        } else {
            throw new IllegalArgumentException("Unsupported template version: must be one of 20, 030, or 035");
        }
    }

    private void writeVersion(DataOutputStream out) throws IOException {
        switch (this.format) {
            case ANSI_378_2004: {
                out.writeByte(32);
                out.writeByte(50);
                out.writeByte(48);
                out.writeByte(0);
                break;
            }
            case ANSI_378_2009: {
                out.writeByte(48);
                out.writeByte(51);
                out.writeByte(48);
                out.writeByte(0);
                break;
            }
            case ANSI_378_2009_AM1: {
                out.writeByte(48);
                out.writeByte(51);
                out.writeByte(53);
                out.writeByte(0);
                break;
            }
            default: {
                throw new IllegalStateException();
            }
        }
    }

    private void readTemplateLength(DataInputStream in) throws IOException {
        if (this.format == ForeignFormat.ANSI_378_2004) {
            int bytes01 = in.readUnsignedShort();
            if (bytes01 < 26) {
                if (bytes01 > 0) {
                    this.format = ForeignFormat.ISO_19794_2_2005;
                    int length = bytes01 << 16 | in.readUnsignedShort();
                    if (length < 24) {
                        logger.warn("Bad template: total length must be at least 24 bytes");
                    }
                } else {
                    int bytes23 = in.readUnsignedShort();
                    if (bytes23 >= 24) {
                        this.format = ForeignFormat.ISO_19794_2_2005;
                    } else {
                        int length = bytes23 << 16 | in.readUnsignedShort();
                        if (length < 26) {
                            logger.warn("Bad template: total length must be at least 26 bytes");
                        } else if (length < 65536) {
                            logger.debug("Not strictly compliant template: 6-byte length field should have value of at least 0x10000");
                        }
                    }
                }
            }
        } else {
            long length = 0xFFFFFFFFL & (long)in.readInt();
            if (this.format == ForeignFormat.ANSI_378_2009 && length < 21L) {
                logger.warn("Bad template: total length must be at least 21 bytes");
            }
            if (this.format == ForeignFormat.ANSI_378_2009_AM1 && length < 40L) {
                logger.warn("Bad template: total length must be at least 40 bytes");
            }
        }
    }

    private void writeTemplateLength(DataOutputStream out) throws IOException {
        int total = this.measure();
        if (this.format != ForeignFormat.ANSI_378_2004) {
            out.writeInt(total);
        } else if (total < 65536) {
            out.writeShort(total);
        } else {
            out.writeShort(0);
            out.writeInt(total);
        }
    }

    private int measure() {
        int fixed;
        switch (this.format) {
            case ANSI_378_2004: {
                fixed = 26;
                break;
            }
            case ANSI_378_2009: 
            case ANSI_378_2009_AM1: {
                fixed = 21;
                break;
            }
            default: {
                throw new IllegalArgumentException();
            }
        }
        int total = fixed + this.fingerprints.stream().mapToInt(f -> f.measure(this.format)).sum();
        if (this.format == ForeignFormat.ANSI_378_2004) {
            return total < 65536 ? total : total + 4;
        }
        return total;
    }

    private void readProductId(DataInputStream in) throws IOException {
        int vendor = in.readUnsignedShort();
        if (vendor == 0) {
            logger.debug("Not strictly compliant template: zero vendor ID / product owner");
        }
        in.skipBytes(2);
    }

    private void writeProductId(DataOutputStream out) throws IOException {
        out.writeShort(259);
        out.writeShort(0);
    }

    private void readSensorInfo(DataInputStream in) throws IOException {
        if (this.format == ForeignFormat.ANSI_378_2004) {
            int combined = in.readUnsignedShort();
            int compliance = combined >> 12;
            if ((compliance & 7) != 0) {
                logger.warn("Bad template: reserved bit in sensor compliance field is set");
            }
        } else {
            int compliance = in.readUnsignedByte();
            if ((compliance & 0x7F) != 0) {
                logger.warn("Bad template: reserved bit in sensor compliance field is set");
            }
            in.skipBytes(2);
        }
    }

    private void writeSensorInfo(DataOutputStream out) throws IOException {
        if (this.format == ForeignFormat.ANSI_378_2004) {
            out.writeShort(0);
        } else {
            out.writeByte(0);
            out.writeShort(0);
        }
    }

    private int readFingerprintCount(DataInputStream in) throws IOException {
        int count = in.readUnsignedByte();
        if ((this.format == ForeignFormat.ANSI_378_2004 || this.format == ForeignFormat.ANSI_378_2009) && count > 176) {
            logger.warn("Bad template: more than 176 fingerprints");
        }
        if (this.format == ForeignFormat.ANSI_378_2009_AM1 && count == 0) {
            logger.warn("Bad template: zero fingerprint count");
        }
        return count;
    }

    private void writeFingerprintCount(DataOutputStream out) throws IOException {
        if (this.fingerprints.size() > 16) {
            throw new IllegalArgumentException("Cannot create template: more than 16 fingerprints, cannot assign view offsets");
        }
        if (this.format == ForeignFormat.ANSI_378_2009_AM1 && this.fingerprints.isEmpty()) {
            throw new IllegalArgumentException("Cannot create template: no fingerprints");
        }
        out.writeByte(this.fingerprints.size());
    }
}

